(ns cider.nrepl.middleware.compile
  "Compiler error and warning information"
  {:author "Jeff Valk"}
  (:require [clojure.tools.nrepl.middleware :refer [set-descriptor!]]
            [clojure.tools.nrepl.middleware.session :refer [session]]
            [clojure.tools.nrepl.misc :refer [response-for]]
            [clojure.tools.nrepl.transport :as t])
  (:import (clojure.lang Compiler$CompilerException)
           (clojure.tools.nrepl.transport Transport)))

;;; ## Wrapping Compilation
;; This middleware wraps the underlying nREPL `eval` handler to provide
;; structured compiler error and warning information. When compiler warnings
;; and/or errors are generated, these are returned as additional messages in the
;; nREPL `eval` response.

;; Ideally this would use compiler hooks, but until `tools.analyzer` replaces
;; the Clojure 1.x compiler (which was definitely not designed with tooling in
;; mind), we have to fetch compiler errors from the environment (as `*e`), and
;; parse warnings from `*err*` output.


;;; ## Compiler Errors/Warnings

;; Instances of `CompilerException` are constructed with source, line, and
;; column info. Bizarrely, however, column isn't stored as a field -- it's just
;; stuffed into the error message, forcing us to parse it out.
(defn compiler-error
  "Return a map describing the compiler error, with the message, file, line,
  and column appended if available."
  [e]
  (let [msg (.getMessage e)]
    (merge
     {:type "error"
      :message msg}
     (when (instance? Compiler$CompilerException e)
       {:file (.source e)
        :line (.line e)
        :column (last (re-find #"compiling:\(.+:\d+:(\d+)\)" msg))}))))

;; Compiler warnings are only available as text, written to `*err*`.
;; The messages emitted are almost entirely consistent in format. The only
;; irregular message is for earmuff naming, which also omits column info.
(defn compiler-warnings
  "Return a sequence of maps describing each compiler warning, with the message,
  file, line, and column appended if available."
  [text]
  (when text
    (for [[msg file line column]
          (concat (re-seq #".* warning, (.+):(\d+):(\d+) - .*" text)
                  (re-seq #"Warning.* change the name. \((.+): *(\d+)\)" text))]
      {:type "warning"
       :message msg
       :file file
       :line line
       :column column})))


;;; ## Middleware

;; BUG :status :eval-error signals that the error has occurred, but there's an
;; inconsistency in nREPL's interruptible-eval handler: the new *e hasn't been
;; assoc'ed to the session when this message is sent. Rather, interruptible-eval
;; assoc's the updated session just after sending the :value or :eval-error
;; message, and prior to :status :done. If we can't fix the nREPL bug, the
;; workaround would be:
;;
;; 1. on :eval-error, assoc a flag to session;
;; 2. on :done, dissoc flag, get error.

(defn wrap-compile
  "Middleware that wraps the underlying `eval` handler to provide structured
  compiler error and warning information. Each compiler error/warning is
  sent as an individual message with slots for `:type` (error or warning),
  `:message`, `:file`, `:line`, and `:column` (when available)."
  [handler]
  (fn [{:keys [op session transport] :as msg}]
    (if (= "eval" op)
      (handler (assoc msg :transport
                      (reify Transport
                        (recv [this] (t/recv transport))
                        (recv [this timeout] (t/recv transport timeout))
                        (send [this {:keys [status err] :as resp}]
                          (when err
                            (doseq [warning (compiler-warnings err)]
                              (t/send transport (response-for msg warning))))

                          (when (:eval-error status)
                            (let [error (compiler-error (@session #'*e))]
                              (t/send transport (response-for msg error))))

                          (t/send transport resp)
                          this))))
      (handler msg))))


;; nREPL middleware descriptor info
(set-descriptor! #'wrap-compile
                 {:requires #{#'session}
                  :expects #{"eval"}})


;;; ## test code
(comment

  (require '[clojure.tools.nrepl.server :as server])
  (require '[clojure.tools.nrepl :as nrepl])

  (server/stop-server server)
  (def server (server/start-server
               :port 7777 :handler (server/default-handler #'wrap-compile)))
  (with-open [transport (nrepl/connect :port 7777)]
    (-> (nrepl/client transport 1000)
        (nrepl/client-session)
        (nrepl/message {:op :eval :code "(def *xxxx* 1) (conj nil 1)"})
        (->> (map nrepl/read-response-value)
             (nrepl/combine-responses))))

  )
