(ns ciir.utils
  ^{:author "David Smith"}
  (:require [clojure.java.io :as jio]
            [clojure.tools.cli :as cli])
  (:import (java.io BufferedReader)
	   (java.util.zip GZIPInputStream GZIPOutputStream)
           (org.apache.commons.compress.compressors.bzip2 BZip2CompressorInputStream)))

(set! *warn-on-reflection* true)

(defn ^BufferedReader gzreader
  [x]
  (-> x jio/input-stream GZIPInputStream. jio/reader))

(defn ^BufferedReader bzreader
  [x]
  (-> x jio/input-stream BZip2CompressorInputStream. jio/reader))

(defn take-to-first
  "Returns a lazy sequence of successive items from coll up to
  and including the point at which it (pred item) returns true.
  pred must be free of side-effects."
  [pred coll]
  (lazy-seq
   (when-let [s (seq coll)]
     (if-not (pred (first s))
       (cons (first s) (take-to-first pred (rest s)))
       (list (first s))))))

(defn partition-when
  "Applies f to each value in coll, splitting it each time f returns
   true. Returns a lazy seq of lazy seqs."
  [f coll]
  (when-let [s (seq coll)]
    (lazy-seq
     (let [run (take-to-first f s)
	   res (drop (count run) s)]
       (cons run (partition-when f res))))))

;; from str-utils2
(defn ^String map-str
  "Apply f to each element of coll, concatenate all results into a
  String."
  [f coll]
  (apply str (map f coll)))

(defn partition-str
  "Splits the string into a lazy sequence of substrings, alternating
  between substrings that match the pattern and the substrings between
  the matches.  The sequence always starts with the substring before
  the first match, or an empty string if the beginning of the string
  matches.

  For example: (partition \"abc123def\" #\"[a-z]+\")
  returns: (\"\" \"abc\" \"123\" \"def\")"
  [^String s ^java.util.regex.Pattern re]
  (let [m (re-matcher re s)]
    ((fn step [prevend]
       (lazy-seq
        (if (.find m)
          (cons (.subSequence s prevend (.start m))
                (cons (re-groups m)
                      (step (+ (.start m) (count (.group m))))))
          (when (< prevend (.length s))
            (list (.subSequence s prevend (.length s)))))))
     0)))

(defn exit
  ([status] (System/exit status))
  ([status msg] (println msg) (System/exit status)))

(defn unindent
  "Unindent multiline string by the first non-zero indent."
  [s]
  (if-let [indent (re-find #"(?m)^[ \t]+" s)]
    (clojure.string/replace s (java.util.regex.Pattern/compile (format "(?m)^%s" indent)) "")
    s))

(defn safe-cli
  [args & specs]
  (try
    (let [[options extra-args banner]
          (apply cli/cli args specs)]
      (when (:help options)
        (exit 0 banner))
      [options extra-args banner])
    (catch Exception e
      (let [[options extra-args banner]
            (apply cli/cli [] specs)]
        (exit 1 (str e "\n" banner))))))
