(ns circle.util.time
  (:refer-clojure :exclude [min max])
  (:require [clj-time.core :as time]
            [clj-time.format :as time-format])
  (:import org.joda.time.DateTime
           (org.joda.time.format DateTimeFormatter)))

(defn java-now
  "Returns a java.util.Time for now"
  []
  (-> (time/now) (.toDate)))

(defn period->str
  "Takes a joda period, returns a human readable string"
  [period]
  (.print (org.joda.time.format.PeriodFormat/getDefault) period))

(defn from-now
  "Takes a period, returns the interval starting from now"
  [period]
  (time/interval (time/now) (time/plus (time/now) period)))

(defn period->interval [period]
  (time/interval (time/now) (time/plus (time/now) period)))

(defn period->secs [p]
  (-> p .toStandardSeconds .getSeconds))

(defn period->millis [p]
  (-> p period->secs (* 1000)))

(defn duration->millis [d]
  (.getMillis d))

(defn interval->millis
  "Takes a joda interval, returns a number of millis in the interval"
  [interval]
  (.toDurationMillis interval))

(defn to-epoch-millis [dt]
  (.getMillis dt))

(defn from-epoch-millis
  "Given a number of millis since the epoch, return a datetime"
  [millis]
  (DateTime. millis))

(def custom-formatters
  {:github-pushed-at (time-format/formatter "yyyy/MM/dd HH:mm:ss Z")})

(defn parse
  "Returns a DateTime instance (in the parsed timezone) time zone obtained by parsing the
   given string according to the given formatter. This function is a copy of clj-time.parse/parse, except that it uses WithOffsetParsed"
  ([#^DateTimeFormatter fmt #^String s]
     (.parseDateTime (.withOffsetParsed fmt) s))
  ([#^String s]
     (first
      (for [f (vals (merge time-format/formatters custom-formatters))
            :let [d (try (parse f s) (catch Exception _ nil))]
            :when d] d))))


(defn pretty-day
  "Given a DateTime, print 'today', 'yesterday', 'monday', 'last week' or 'may 25th'"
  [date]
  (condp > (-> date (time/interval (time/now)) time/in-days)
    1 "today"
    2 "yesterday"
    7 (-> date .dayOfWeek .getAsText)
    (str date)))

(defn destruct
  "Returns a vector, the parts of the time, that can be applied to the date-time constructor"
  [dt]
  [(time/year dt)
   (time/month dt)
   (time/day dt)
   (time/hour dt)
   (time/minute dt)
   (time/sec dt)
   (time/milli dt)])

(def time-length-map
  {:year 1
   :month 2
   :day 3
   :hour 4
   :minute 5
   :second 6
   :milli 7})

(defn round
  "Truncate the datetime to the nearest period. period can be :year :month :date :hour :second"
  [dt period]
  (->> dt
      (destruct)
      (take (time-length-map period))
      (apply time/date-time)))

(defn duration
  "Returns the duration between two datetimes"
  [dt-a dt-b]
  (.toDuration (time/interval dt-a dt-b)))

(defn interval->duration [i]
  (.toDuration i))

(defn duration-millis [d]
  (.getMillis d))

(defn duration>
  "True if this duration is longer than Period p"
  [d p]
  (> (duration-millis d)
     (-> p period->millis)))

(defn duration< [d p]
  (not (duration> d p)))

(defn max
  "Returns the greater of the two datetimes"
  [dt-a dt-b]
  (if (time/after? dt-b dt-a)
    dt-b
    dt-a))

(defn maybe-interval
  "Returns an interval, or nil when dt-b is before dt-a. Use when the two datetimes could be created by different machines and clock skew, or bugs, or races"
  [dt-a dt-b]
  (when (time/after? dt-b dt-a)
    (time/interval dt-a dt-b)))
