
(ns cirru-editor.comp.expression
  (:require [hsl.core :refer [hsl]]
            [respo.alias :refer [create-comp div]]
            [respo.comp.debug :refer [comp-debug]]
            [cirru-editor.comp.token :refer [comp-token]]
            [cirru-editor.util.detect :refer [coord-contains?
                                              shallow?
                                              deep?]]
            [cirru-editor.util.keycode :as keycode]))

(def style-expression
 {:border-style "solid",
  :min-width "16px",
  :box-sizing "border-box",
  :vertical-align "top",
  :min-height "26px",
  :margin-left 12,
  :margin-top 0,
  :padding-right 0,
  :border-width "0 0 0 1px",
  :padding-top 2,
  :padding-left 8,
  :outline "none",
  :border-color (hsl 0 0 100 0.24),
  :padding-bottom 0,
  :margin-right 0,
  :margin-bottom 4})

(declare comp-expression)

(defn on-click [modify! coord focus]
  (fn [e dispatch!]
    (if (not= coord focus) (modify! :focus-to coord dispatch!))))

(defn on-keydown [modify! coord on-command]
  (fn [e dispatch!]
    (let [code (:key-code e)
          event (:original-event e)
          shift? (.-shiftKey event)
          command? (or (.-metaKey event) (.-ctrlKey event))]
      (cond
        (= code keycode/space) (do
                                 (.preventDefault event)
                                 (if
                                   shift?
                                   (modify!
                                     :before-token
                                     coord
                                     dispatch!)
                                   (modify!
                                     :after-token
                                     coord
                                     dispatch!)))
        (= code keycode/tab) (do
                               (.preventDefault event)
                               (if
                                 shift?
                                 (modify!
                                   :unfold-expression
                                   coord
                                   dispatch!)
                                 (modify!
                                   :fold-node
                                   coord
                                   dispatch!
                                   dispatch!)))
        (= code keycode/enter) (if
                                 command?
                                 (if
                                   shift?
                                   (modify!
                                     :prepend-expression
                                     coord
                                     dispatch!)
                                   (modify!
                                     :append-expression
                                     coord
                                     dispatch!))
                                 (if
                                   shift?
                                   (modify!
                                     :before-expression
                                     coord
                                     dispatch!)
                                   (modify!
                                     :after-expression
                                     coord
                                     dispatch!)))
        (= code keycode/backspace) (do
                                     (.preventDefault event)
                                     (modify!
                                       :remove-node
                                       coord
                                       dispatch!))
        (= code keycode/left) (do
                                (.preventDefault event)
                                (modify! :node-left coord dispatch!))
        (= code keycode/right) (do
                                 (.preventDefault event)
                                 (modify! :node-right coord dispatch!))
        (= code keycode/up) (do
                              (.preventDefault event)
                              (modify! :node-up coord dispatch!))
        (= code keycode/down) (do
                                (.preventDefault event)
                                (modify!
                                  :expression-down
                                  coord
                                  dispatch!))
        (and command? (= code keycode/key-b)) (do
                                                (.preventDefault event)
                                                (modify!
                                                  :duplicate-expression
                                                  coord
                                                  dispatch!))
        (and command? (= code keycode/key-c)) (modify!
                                                :command-copy
                                                coord
                                                dispatch!)
        (and command? (= code keycode/key-x)) (modify!
                                                :command-cut
                                                coord
                                                dispatch!)
        (and command? (= code keycode/key-v)) (modify!
                                                :command-paste
                                                coord
                                                dispatch!)
        :else (if command? (on-command e dispatch!) nil)))))

(defn render [expression
              modify!
              coord
              level
              tail?
              focus
              on-command
              head?
              after-expression?]
  (fn [state mutate!]
    (let [exp-size (count expression)]
      (div
        {:style
         (merge
           style-expression
           (if (and
                 (shallow? expression)
                 (not after-expression?)
                 (not tail?)
                 (not head?)
                 (pos? level)
                 (< (count expression) 5))
             {:text-align "center",
              :margin-left 4,
              :background-color (hsl 200 80 80 0),
              :padding-right 15,
              :border-width "0 0 1px 0",
              :padding-left 17,
              :display "inline-block",
              :padding-bottom 2,
              :margin-right 4})
           (if (and tail? (not head?) (pos? level))
             {:background-color (hsl 0 80 80 0),
              :border-width "0 0 0 1px",
              :display "inline-block"})
           (if (= coord focus) {:border-color (hsl 0 0 100 0.5)})),
         :event
         {:keydown (on-keydown modify! coord on-command),
          :click (on-click modify! coord focus)},
         :attrs
         (merge
           {:tab-index 0}
           (if (= coord focus) {:id "editor-focused"}))}
        (loop [acc []
               idx 0
               expr expression
               child-after-expression? false]
          (if (empty? expr)
            acc
            (let [item (first expr)
                  pair [idx
                        (let [child-coord (conj coord idx)
                              child-focus (if
                                            (coord-contains?
                                              focus
                                              child-coord)
                                            focus
                                            nil)
                              child-head? (zero? idx)]
                          (if (string? item)
                            (comp-token
                              item
                              modify!
                              child-coord
                              child-focus
                              on-command
                              child-head?)
                            (comp-expression
                              item
                              modify!
                              child-coord
                              (inc level)
                              (and
                                (or after-expression? (not tail?))
                                (= (dec exp-size) idx))
                              child-focus
                              on-command
                              child-head?
                              child-after-expression?)))]
                  next-acc (conj acc pair)]
              (recur
                next-acc
                (inc idx)
                (rest expr)
                (vector? item)))))))))

(def comp-expression (create-comp :expression render))
