(ns pedestal-utils.auth
  (:require [utils.employee-info-service :as employee]
            [io.pedestal.interceptor.chain :refer [terminate enqueue*]]
            [utils.mdc :as MDC]))

(defn remove-request-logging-fields []
  (MDC/remove-tenant-id) (MDC/remove-user-id) (MDC/remove-request-id))

(defn get-token [headers]
  (let [auth-header (get headers "authorization")
        bearer-length (count "Bearer ")]
    (if (not (nil? auth-header))
      (if (< bearer-length (count auth-header))
        (subs auth-header bearer-length)
        "")
      "")))

(def pedestal-401-response
  {:name  :401-response
   :leave (fn [context]
            (assoc context :response {:status  401
                                      :body    {:userMessage "Unauthorized"}}))})

(def authenticate
  {:name  :auth-interceptor
   :enter (fn [{{headers :headers} :request :as context}]
            (let [token (get-token headers)
                  {{employee-id :id
                    tenant-id   :tenantId :as employee-info} :body
                   status                                    :status} (employee/do-remote-employee-info-call token)]
              (if (= status 200)
                (do (MDC/put-tenant-id tenant-id) (MDC/put-request-id (get headers "x-request-id")) (MDC/put-user-id employee-id)
                    (assoc-in context [:request :employee-info] employee-info))
                (-> context
                    terminate
                    (enqueue* pedestal-401-response)))))
   :leave (fn [context]
            (remove-request-logging-fields)
            context)})