(ns utils.rabbit-service
  (:require [langohr.core :as rmq]
            [langohr.channel :as lch]
            [langohr.queue :as lq]
            [langohr.basic :as lb]
            [langohr.consumers :as lc]
            [clojure.data.json :as json]
            [utils.logger :as logger]))

(def ^{:const true} default-exchange-name "")

(defn connection-settings [host-env username-env password-env]
  {:username (System/getenv username-env)
   :password (System/getenv password-env)
   :vhost "/"
   :host (System/getenv host-env)
   :port 5672})

(defn message-handler-wrapper [raw-handler]
  (fn [ch {:keys [content-type delivery-tag type] :as meta} payload]
    (raw-handler (json/read-str (String. payload "UTF-8") :key-fn keyword))))

(defn start-consumer
  [queue-name message-handler]
  (try
    (let [conn (rmq/connect (connection-settings "AMQPHOST" "AMQPUSER" "AMQPPASS"))
          ch (lch/open conn)
          callback (message-handler-wrapper message-handler)]
      (lq/declare ch queue-name {:exclusive false :auto-delete false :durable true})
      (lc/subscribe ch queue-name callback {:auto-ack true}))
    (catch Throwable th (logger/debug th))))

(defn publish
  [message queue-name]
  (try
    (let [conn (rmq/connect (connection-settings "AMQPHOST" "AMQPUSER" "AMQPPASS"))
          ch (lch/open conn)]
      (lq/declare ch queue-name {:auto-delete false :durable true :exclusive false})
      (lb/publish ch default-exchange-name queue-name message)
      (Thread/sleep 100)
      (rmq/close ch)
      (rmq/close conn))
    (catch Throwable th (logger/debug th))))
  
(defn publish->exchange
  ([type exchange-name routing-key message]
   (publish->exchange type exchange-name routing-key message {:content-type "application/json"}))
  ([type exchange-name routing-key message headers]
   (try
     (let [conn (rmq/connect (q-connection-settings))
           ch (lch/open conn)]
       (le/declare ch exchange-name type {:auto-delete false :durable true :exclusive false})
       (lb/publish ch exchange-name routing-key (json/generate-string message) headers)
       (Thread/sleep 100)
       (rmq/close ch)
       (rmq/close conn))
     (catch Throwable th (logger/error th)))))
