(ns utils.rabbit-service
  (:require [langohr.core :as rmq]
            [langohr.channel :as lch]
            [langohr.queue :as lq]
            [langohr.basic :as lb]
            [langohr.consumers :as lc]
            [langohr.exchange :as le]
            [cheshire.core :as json]
            [utils.logger :as logger]))

(def ^{:const true} default-exchange-name "")

(defn connection-settings [host-env username-env password-env]
  {:username (System/getenv username-env)
   :password (System/getenv password-env)
   :vhost    "/"
   :host     (System/getenv host-env)
   :port     5672})

(def q-connection-setting (connection-settings "QAMQPHOST" "AMQPUSER" "AMQPPASS"))

(defn parse-rabbit-payload [payload]
  (json/parse-string (String. payload "UTF-8") true))

(defn message-handler-wrapper [raw-handler ch {:keys [content-type delivery-tag type routing-key] :as meta} payload]
  (raw-handler (parse-rabbit-payload payload)))


(defn start-consumer
  [queue-name message-handler]
  (try
    (let [conn (rmq/connect (connection-settings "AMQPHOST" "AMQPUSER" "AMQPPASS"))
          ch (lch/open conn)
          callback (partial message-handler-wrapper message-handler)]
      (lq/declare ch queue-name {:exclusive false :auto-delete false :durable true})
      (lc/subscribe ch queue-name callback {:auto-ack true}))
    (catch Throwable th (logger/debug th))))

(defn publish
  [message queue-name]
  (try
    (let [conn (rmq/connect (connection-settings "AMQPHOST" "AMQPUSER" "AMQPPASS"))
          ch (lch/open conn)]
      (lq/declare ch queue-name {:auto-delete false :durable true :exclusive false})
      (lb/publish ch default-exchange-name queue-name message)
      (Thread/sleep 100)
      (rmq/close ch)
      (rmq/close conn))
    (catch Throwable th (logger/debug th))))

(defn publish->exchange
  ([type exchange-name routing-key message]
   (publish->exchange type exchange-name routing-key message {:content-type "application/json"}))
  ([type exchange-name routing-key message headers]
   (try
     (let [conn (rmq/connect (connection-settings "QAMQPHOST" "AMQPUSER" "AMQPPASS"))
           ch (lch/open conn)]
       (le/declare ch exchange-name type {:auto-delete false :durable true :exclusive false})
       (lb/publish ch exchange-name routing-key (json/generate-string message) headers)
       (Thread/sleep 100)
       (rmq/close ch)
       (rmq/close conn))
     (catch Throwable th (logger/error th)))))

(defn bind->exchange
  "when passing a routing information for a routing or topic exchange it should be a map with one or more key value pairs:
  i.e. {'orange-route' (fn [...] (do-something. . .)) }"
  [exchange-name type & {:keys [queue-name topics handler] :or {queue-name "" topics [""]}}]
  (try
    (let [conn (rmq/connect q-connection-setting)
          ch (lch/open conn)
          queue-name' (:queue (lq/declare ch queue-name {:auto-delete false :durable true :exclusive true}))]
      (le/declare ch exchange-name type {:auto-delete false :durable true :exclusive false})
      (doseq [topic topics]
        (lq/bind ch queue-name' exchange-name {:routing-key topic}))
      (lc/subscribe ch queue-name' handler))
    (catch Throwable th (logger/error th))))