(ns utils.health-service
  (:require [utils.mysql-service :as mysql]
            [rethinkdb.core :as rethink]
            [langohr.core :as rabbit]
            [clojure.java.io :as io]
            [clj-http.client :as http])
  (:import java.io.File))

(def elasticsearch-host "https://search-clientsuccess-uktboeyl3lzmax3s2gihggksky.us-west-2.es.amazonaws.com/")

(defn connection-settings []
  {:username (System/getenv "AMQPUSER")
   :password (System/getenv "AMQPPASS")
   :vhost    "/"
   :host     (System/getenv "AMPQPHOST")
   :port     5672})

(defn check-mysql-connection []
  (try (mysql/run-query "select 1 + 1" [])
       "UP"
       (catch Exception e
         "DOWN")))

(defn check-rethink-connection []
  (try (rethink/connect :host (System/getenv "RDBHOST") :port 28015 :db-name nil)
       "UP"
       (catch Exception e
         "DOWN")))

(defn check-rabbit-connection []
  (try (rabbit/connect (connection-settings))
       "UP"
       (catch Exception e
         "DOWN")))

(defn check-diskspace-status
  ([total free]
   (check-diskspace-status total free 0.90))
  ([total free threshold]
   (if (<= (float (/ (- total free) total)) threshold)
     "UP"
     "DOWN")))

(defn get-disk-space [threshold]                            ;threshold as a percent of total
  (let [file (io/file "/")
        total (.getTotalSpace file)
        free (.getFreeSpace file)]
    {:status (check-diskspace-status total free threshold)
     :total  total
     :free   free}))

(defn get-environment []
  (System/getenv "ENVIRONMENT"))

(defn elastic-search-request []
  (let [env (get-environment)]
    (http/get (str elasticsearch-host "/" env "_contacts," env "_clients/_search?")
              {:client-params    {:cookie-policy (constantly nil)}
               :content-type     :json
               :throw-exceptions false
               :as               :json
               :insecure?        true})))

(defn get-elastic-search-connection []
  (if (= (:status (elastic-search-request) 200))
    "UP"
    "DOWN"))

(defn health [& args]
  {:status 200
   :body   {:status        "UP"
            :db            [{:database "mysql"
                             :status   (check-mysql-connection)}
                            {:database "rethink"
                             :status   (check-rethink-connection)}]
            :rabbit        (check-rabbit-connection)
            :diskSpace     (get-disk-space 0.90)
            :elasticSearch (get-elastic-search-connection)}})
