(ns io.datafy.http.server.stream-handler
  (:require [clojure.tools.logging :as log]
            [cheshire.core :as json]
            [clojure.data.csv :as csv]
            [io.datafy.http.api :as api]
            [io.datafy.http.component.rdf.rdf :as rdf]
            [io.datafy.http.server.util :as w]
            [io.datafy.http.core.spec2 :as cs])
  (:import [javax.servlet ServletOutputStream]
           [java.io OutputStreamWriter BufferedWriter]))

(defmulti stream-handler (fn [_ r] (get-in r [::cs/extension])))

(defmethod stream-handler
  :default
  [db req]
  (let [r (api/stream db (:ds-name req) (:payload req) (:meta-params req) )
        out-name (or (::cs/entity-name req)
                     (::cs/ds-name req))
        out-name (name out-name)]
    {:status  200
     :headers {"Content-Type"        "application/octet-stream; charset=utf-8"
               "Content-Disposition" (str "attachment; fileName=" out-name ".json")}
     :body    (fn [^ServletOutputStream stream-obj]
                (let [handler (fn [v]
                                (let [v (dissoc v :_graph_ :_graph_spec)
                                      [k v] (first v)]
                                  (doseq [v1 v]
                                    (->> v1
                                         (json/generate-string)
                                         (.print stream-obj)))))]
                  (r handler)))}))


(defmethod stream-handler
  :rdf
  [db req]
  (let [req (assoc-in req [::cs/meta-params ::cs/result-format] :graph)
        repo (api/stream db (:ds-name req) (:payload req) (:meta-params req) )
        out-name (or (::cs/entity-name req)
                     (::cs/ds-name req))
        out-name-str (name out-name)]
    {:status  200
     :headers {"Content-Type"        "application/octet-stream; charset=utf-8"
               "Content-Disposition" (str "attachment; fileName=" out-name-str ".ttl")}
     :body    (fn [^ServletOutputStream stream-obj]
                (let [handler (fn [m]
                                ;(clojure.pprint/pprint m)
                                (let [m (if (sequential? m) {out-name m} m)]
                                  (doseq [[k v] m
                                          v (partition-all 10 v)]
                                    (let [v1 (rdf/as-rdf (assoc {} k v :_graph_spec (get m :_graph_spec)))]
                                      (.print stream-obj v1)))
                                  (.print stream-obj (rdf/as-rdf (select-keys m [:_graph_])))
                                  ))]
                  (repo handler)))}))


(defmethod stream-handler
  :edn
  [db req]
  (let [repo (api/stream db (:ds-name req) (:payload req) (:meta-params req) )
        out-name (or (::cs/entity-name req)
                     (::cs/ds-name req))
        out-name (name out-name)]
    {:status  200
     :headers {"Content-Type"        "application/octet-stream; charset=utf-8"
               "Content-Disposition" (str "attachment; fileName=" out-name ".edn")}
     :body    (fn [^ServletOutputStream stream-obj]
                (let [handler (fn [v]
                                (let [v (dissoc v :_graph_ :_graph_spec)
                                      [k v] (first v)]
                                  (doseq [v1 v]
                                    (->> (pr-str v1)
                                         (.print stream-obj)))))]
                  (repo handler)))}))


(defmethod stream-handler
  :csv
  [db req]
  (let [repo (api/stream db (:ds-name req) (:payload req) (:meta-params req) )
        sim (get-in req [::cs/meta-params ::cs/sim])
        out-name (or (::cs/entity-name req)
                     (::cs/ds-name req))
        out-name (name out-name)]
    {:status  200
     :headers {"Content-Type"        "text/csv; charset=utf-8"
               "Content-Disposition" (str "attachment; fileName=" out-name ".csv")}
     :body    (fn [^ServletOutputStream stream-obj]
                (let [header (volatile! nil)]
                  (with-open [o (BufferedWriter. (OutputStreamWriter. stream-obj))]
                    (let [handler (fn [v]
                                    (let [v (dissoc v :_graph_ :_graph_spec)
                                          [k v] (first v)
                                          v (if sim (mapv w/to-simple-flatten-map v) v)]
                                      (when-not @header
                                        (vreset! header (keys (first v)))
                                        (csv/write-csv o (vector (mapv name @header))))
                                      (csv/write-csv o (mapv (apply juxt @header) v))))]
                      (repo handler)))))}))


