(ns io.datafy.http.server.pedestal-interceptor
  (:require [io.pedestal.http :as http]
            [io.pedestal.http.content-negotiation :as conneg]
            [io.pedestal.http.ring-middlewares :as ring-mw]
            [clojure.tools.logging :as log]
            [io.pedestal.http.body-params :as body-params]
            [io.pedestal.http.ring-middlewares :as middlewares]
            [ring.middleware.session.cookie :as cookie]
           ; [io.datafy.core.spec2 :as cs]
           ; [io.datafy.core.error :as ce]
            [io.datafy.http.server.req-parser :as rp]
            [io.pedestal.interceptor.error :as err]))

(defn response [status body & {:as headers}]
  {:status status :body body :headers headers})

(def ok (partial response 200))
(def created (partial response 201))
(def accepted (partial response 202))
(def bad-request (partial response 400))
(def error (partial response 500))

(def default-serialisation-m
  (hash-map
    "application/json" http/json-body
    "application/transit+json" http/transit-json-body
    "application/transit+msgpack" http/transit-msgpack-body
    "application/transit" http/transit-body))

(defn get-accept-type [request]
  (let [accept-type (or
                      (get-in request [:accept :field])
                      (get-in request [:headers "content-type"]))]
    (if (or (nil? accept-type)
            (clojure.string/blank? accept-type))
      "application/json"
      accept-type)))

(def data-serialisation-interceptor
  {:name  :format-response-interceptor
   :leave (fn [context]
            (let [accept-type2 (get-accept-type (:request context))
                  l-fn (if-let [v (get default-serialisation-m accept-type2)]
                         (:leave v)
                         (:leave http/json-body))]
              (l-fn context)))})

(def warp-response-with-data-interceptor
  {:name  :api-handler
   :leave (fn [context]
            (let [m (rp/parse-repo-path (:request context))]
              ;(log/debug "extenstion " (get-in m [:extension]))
              (if (get-in m [:extension])
                context
                (->> (ok (:response context))
                     (assoc context :response)))))})

(def warp-response-with-error-interceptor
  (err/error-dispatch
    [ctx ex]
    [{:exception-type :clojure.lang.ExceptionInfo}]
    (let [
          e-data (-> (ex-data ex)
                     (get :exception)
                     (ex-data)
                     #_(assoc :http-request r))
          _ (clojure.pprint/pprint e-data)
          error (if (and (map? e-data)
                         (get e-data :error)
                         )
                  e-data
                  {:error e-data}
                  )  #_{:error e-data}]

      (log/info "Request processing error for " #_(clojure.datafy/datafy ex) (ex-data ex))

      (condp = (get-in e-data [:error-code])
        404 (do
              (assoc ctx :response (response 404 error)))
        400 (do
              (log/info "Request processing error for " error)
              (assoc ctx :response (bad-request error)))
        (do
          (assoc ctx :response (bad-request error)))))
    :else
    (let [e-data {:description "Internal error, check server log "}
          e-res (->> (select-keys (get-in ctx [:request]) [:ds-name :entity-name :attr-filter :meta-params])
                     (reduce-kv (fn [acc k v]
                                  (assoc acc (name k) v)) {})
                     (assoc e-data :http-request))
          error {:error e-res}]
      ;(log/error (clojure.datafy/datafy ex))
      (assoc ctx :response (bad-request error)))))

(defn throw-for-invalid-content [content-type supported-type]
  (throw (ex-info "Invalid content "
                  {:for        :content-type
                   :provided   content-type
                   :expected   (clojure.string/join "," (mapv name supported-type))
                   :msg        "Invalid content "
                   :error-code bad-request})))

(def validate-content-type-interceptor
  {:name  :validate-content-type-interceptor
   :enter (fn [context]
            (when (= :post (get-in context [:request :request-method]))
              (let [s-content #{"application/x-www-form-urlencoded"
                                "application/json"
                                "application/edn"
                                "application/text"
                                "text/plain"
                                "application/transit+json"
                                "application/transit+msgpack"
                                "application/transit"}]
                (when-not (contains? s-content (get-in context [:request :headers "content-type"]))
                  (throw-for-invalid-content (get-in context [:request :content-type]) s-content))))
            context)})

(def parse-content-type-interceptor (conneg/negotiate-content ["application/x-www-form-urlencoded"
                                                               "application/edn"
                                                               "application/json"
                                                               "application/text"
                                                               "text/plain"
                                                               "application/transit+json"
                                                               "application/transit+msgpack"
                                                               "application/transit"]))

#_(def parse-request-inter
  {:name  :parse-request
   :enter (fn [context]
            (let [request (:request context)
                  body-param (if (get-in request [:path-params :api])
                               (rp/parse-post-request request)
                               (rp/api-request-parse (or (:transit-params request)
                                                         (:edn-params request)
                                                         (:json-params request))))
                  request (assoc request :body-params body-param)]
              (assoc context :request request)))})

(def data-interceptors
  (let [session-interceptor (middlewares/session {:store (cookie/cookie-store)})]
    [data-serialisation-interceptor
     warp-response-with-error-interceptor
     warp-response-with-data-interceptor
     parse-content-type-interceptor
     validate-content-type-interceptor
     (body-params/body-params)
     session-interceptor]
    )
  )


(def upload-interceptor
  [data-serialisation-interceptor
   warp-response-with-error-interceptor
   warp-response-with-data-interceptor
  ; parse-content-type-interceptor
 ;  validate-content-type-interceptor
   ;(body-params/body-params)
   (ring-mw/multipart-params)
   ]

  )


(defn html-interceptors []
  (let [session-interceptor (middlewares/session {:store (cookie/cookie-store)})]
    [data-serialisation-interceptor
     ;warp-response-with-error-interceptor
     ;warp-response-with-data-interceptor
     parse-content-type-interceptor
     validate-content-type-interceptor
     (body-params/body-params)
     session-interceptor]
    )
  )