(ns io.datafy.http.server.req-parser
  (:require [clojure.tools.logging :as log]
            [io.datafy.http.server.util :as u]
            [clojure.set :as set]
            [io.datafy.http.server.error :as ce]))

(defn as-int [v]
  (try
    (if (string? v)
      (Integer/parseInt v)
      v)
    (catch Exception e
      (log/debug e))))

(defn get-meta-param [m]
  (let [only-v (get m nil)
        mk (if (sequential? only-v)
             (into {} (comp (map keyword) (map (fn [v] {v nil}))) only-v)
             {(keyword only-v) nil})
        mk (dissoc mk nil)
        m (dissoc m nil)
        xf (comp (map name)
                 (filter (fn [v] (clojure.string/starts-with? v "_")))
                 (map keyword))
        meta-key (into [] xf (keys m))
        w (select-keys m meta-key)
        w (if-let [p (get w :_total)]
            (assoc w :_total (Integer/parseInt p))
            w)
        w (if-let [p (get w :_page)]
            (assoc w :_page (Integer/parseInt p))
            w)
        w (if-let [p (get w :_psize)]
            (assoc w :_psize (Integer/parseInt p))
            w)]
    (merge mk w)))

(defn as-repo [coll]
  (keyword (clojure.string/join "." coll)))

(defn get-ds-name [request]
  (let [w (get-in request [:path-params :repo])
        w (first (clojure.string/split w #"\."))]
    (->> (clojure.string/split w #"/")
         (remove clojure.string/blank?)
         (into []))))

(defn parse-repo-path [request]
  (when (get-in request [:path-params :repo])
    (let [repo-param-str (get-in request [:path-params :repo])
          [ds-name-str ext] (clojure.string/split repo-param-str #"\.")
          repository-name (->> (clojure.string/split ds-name-str #"/")
                               (remove clojure.string/blank?)
                               (into []))
          [ds-name iden-name] (if (< 1 (count repository-name))
                                [(butlast repository-name) (last repository-name)]
                                [repository-name nil])
          ds-name (keyword (clojure.string/join "." ds-name))
          ident-name (when iden-name (keyword (clojure.string/lower-case iden-name)))
          ext (when ext (keyword ext))]
      (-> {}
          (assoc :repo-path (keyword (clojure.string/join "." repository-name)))
          (assoc :ds-name ds-name)
          (assoc :extension ext)
          (assoc :entity-name ident-name)))))


(defn parse-query-params [m request]
  (let [ident-name (:entity-name m)
        params (:params request)
        meta-params (get-meta-param params)
        p (apply dissoc params (keys meta-params))

        meta-params (clojure.set/rename-keys meta-params {:_filter :filter
                                                          :_total  :total
                                                          ;:_flat   :flat
                                                          :_remove :remove
                                                          ;:_sim    :sim
                                                          :_format :result-format
                                                          :_psize  :page-size
                                                          :_page   :page})
        meta-params (if (get meta-params :result-format)
                      (update meta-params :result-format keyword)
                      meta-params)
        p (dissoc p nil)
        params (u/from-flatten-map p)
        payload (if ident-name
                 (if params
                   {ident-name params}
                   {ident-name {}})
                 {})]
    (-> m
        (assoc :req-source :api)
        (assoc :payload payload)
        (assoc :attr-filter (or p {}))
        (assoc :meta-params meta-params))))


(defn validate-request! [request]
  (when-not (map? request)
    (ce/throw-for-invalid-api-request request))
  (when-let [w (:meta request)]
    (when-not (map? w)
      (ce/throw-for-invalid-meta-request w))

    (when-not (set/superset? #{:remove :filter :flat :sim :page :psize}
                             (into #{} (keys w)))
      (ce/throw-for-invalid-meta-request w))))

(defn api-request-parse [request]
  (validate-request! request)
  (let [meta-param (get request :meta)
        meta-param (cond-> meta-param
                           (get meta-param :page) (update :page as-int)
                           (get meta-param :psize) (update :psize as-int)
                           (= 2 2) (clojure.set/rename-keys {:filter        :filter
                                                             :result-format :result-format
                                                             ;:flat   :flat
                                                             :remove        :remove
                                                             ;:sim    :sim
                                                             :page          :page
                                                             :psize         :psize}))]
    (-> request
        (assoc :meta-params meta-param))))

(defn debug [v]
  (log/debug v)
  v)


(defn parse-post-request
  [request]
  (log/debug "Parse request for http ")
  (condp = (:request-method request)
    :get (-> (parse-repo-path request)
             (parse-query-params request))
    :delete (-> (parse-repo-path request)
                (parse-query-params request))
    :post
    (if-not (= "application/x-www-form-urlencoded" (get-in request [:headers "content-type"]))
      (-> (parse-repo-path request)
          (parse-query-params request))
      (let [repository-name (get-ds-name request)
            e-name (keyword (last repository-name))
            ds-name (as-repo (butlast repository-name))
            params (:form-params request)
            params (u/from-flatten-map params)
            body-param {e-name params}]
        (->> {:api         :push
              :ds-name     ds-name
              :payload     body-param
              :entity-name e-name
              :attr-filter (or params {})})))
    (ce/throw-for-invalid-api-request (select-keys request [:path-params :request-method]))))


(defn as-request-api [request]
  (let [rm (:request-method request)]
    (condp = rm
      :get
      (-> (parse-repo-path request)
          (parse-query-params request)
          (assoc :api :pull))
      :delete
      (-> (parse-repo-path request)
          (parse-query-params request)
          (assoc :api :delete))
      :post
      (-> (parse-post-request request)
          (assoc :api :push)))))