(ns ksql.gen.connector.config-generator
  (:require [cheshire.core :as json]
            [ksql.gen.reader.csv-data-reader :as r]
            [ksql.gen.protocol :as p]))


(defn as-struct-value-schema [{:keys [fields key]} n]
  (let [fm (into {}
                 (comp
                   (filter (fn [r]
                             (not (contains? #{"ROWTIME" "ROWKEY"} (get r :name)))))
                   (map (fn [r]
                          (let [v (-> (get r :schema)
                                      (select-keys [:type])
                                      (update :type clojure.string/upper-case)
                                      (assoc :isOptional "true"))
                                k (keyword (clojure.string/lower-case (get r :name)))]
                            {k v}))))
                 fields)]
    {:type         "STRUCT",
     :name         (str "io.datafy.connector.kafka.ksql." (clojure.string/lower-case n))
     :isOptional   "false",
     :fieldSchemas fm}))


(defn as-struct-key-schema [{:keys [fields key]} n]
  (let [fm (into {}
                 (comp
                   (filter (fn [r]
                             ;(println "--r" r)
                             (= (get r :name) key)))
                   (map (fn [r]
                          (let [v (-> (get r :schema)
                                      (select-keys [:type])
                                      (update :type clojure.string/upper-case)
                                      (assoc :isOptional "true"))
                                ;   _ (println "--"v)
                                k (keyword (clojure.string/lower-case (get r :name)))]
                            {k v}))))
                 fields)]
    {:type         "STRUCT",
     :name         (str "io.datafy.connector.kafka.ksql." (clojure.string/lower-case n))
     :isOptional   "false",
     :fieldSchemas fm}))



(defmethod p/gen-connector
  "com.github.jcustenborder.kafka.connect.spooldir.SpoolDirJsonSourceConnector"
  [connector-schema]
  ; (  connector-schema)
  (let [f-name (get connector-schema :name)
        n (clojure.string/upper-case f-name)                ;(str  "_STAGING")
        ;ref-name (get connector-schema :connector_ref_name)
        template-v (get connector-schema :template )
        template-name (get connector-schema :template-file-name)
        ;  ref-name (or ref-name target-name)
        file-path (get connector-schema :file-path)
        file-name (r/get-file-name file-path)
        ]
    (-> template-v
        (assoc :name (clojure.string/lower-case (str n "_" template-name)))
        (assoc-in [:config :topic] (clojure.string/lower-case n))
        (assoc-in [:config :input.file.pattern] (str "^" file-name ".*.json")))))



(defmethod p/gen-connector
  "com.github.jcustenborder.kafka.connect.spooldir.SpoolDirCsvSourceConnector"
  [connector-schema]
 ; (  connector-schema)
  (let [f-name (get connector-schema :name)
        n (clojure.string/upper-case f-name)                ;(str  "_STAGING")
        value-schema (as-struct-value-schema connector-schema n)
        key-schema (as-struct-key-schema connector-schema n)
        ;ref-name (get connector-schema :connector_ref_name)
        template-v (get connector-schema :template )
        template-name (get connector-schema :template-file-name)
        target-name (->> (clojure.string/split f-name #"_")
                         (butlast)
                         (clojure.string/join "_"))
        ;  ref-name (or ref-name target-name)
        file-path (get connector-schema :file-path)
        file-name (r/get-file-name file-path)
        ]
    (-> template-v
        (assoc :name (clojure.string/lower-case (str n "_" template-name)))
        (assoc-in [:config :topic] (clojure.string/lower-case n))
        (assoc-in [:config :value.schema] (json/generate-string value-schema))
        (assoc-in [:config :key.schema] (json/generate-string key-schema))
        (assoc-in [:config :input.file.pattern] (str "^" file-name ".*.csv$")))))



(defmethod p/gen-connector
  "io.confluent.connect.jdbc.JdbcSinkConnector"
  [connector-schema]
  ;(  connector-schema)
  (let [f-name (get connector-schema :entity_name)
        ref-name (get connector-schema :ref_name)
        ref-name (if (or (nil? ref-name)
                         (clojure.string/blank? ref-name))
                   nil
                   ref-name)
        topic-name (clojure.string/lower-case (or (get connector-schema :topic) f-name) ) #_(clojure.string/upper-case f-name)


        table-name (or ref-name f-name)
        ref-name (if-let [w ref-name]
                   (str f-name "_" w)
                   f-name)
        template-v (get connector-schema :template )
        template-name (get connector-schema :template-file-name)]
    (-> template-v
        (assoc :name (clojure.string/lower-case (str ref-name "_" template-name)))
        (assoc-in [:config :topics] topic-name)
        (update-in [:config :pk.fields] (fn [v]
                                         ; (clojure.pprint/pprint connector-schema)
                                          (if (= v "none")
                                            v
                                            (when-let [k (get connector-schema :key)]
                                              (clojure.string/upper-case  k)

                                              )
                                            )
                                          ))
        (assoc-in [:config :table.name.format] table-name))))


(defmethod p/gen-connector
  "io.confluent.connect.elasticsearch.ElasticsearchSinkConnector"
  [connector-schema]
  ;(println schema)
  (let [f-name (get connector-schema :entity_name)
        ref-name (get connector-schema :ref_name)
        ref-name (if (or (nil? ref-name)
                         (clojure.string/blank? ref-name))
                   nil
                   ref-name)
        topic-name (clojure.string/lower-case f-name)

        ref-name (if-let [w ref-name]
                   (str f-name "_" w)
                   f-name)

        index-name (or ref-name topic-name)
        template-v (get connector-schema :template )
        template-name (get connector-schema :template-file-name)
        ]

    (-> template-v
        (assoc :name (clojure.string/lower-case (str ref-name "_" template-name)))
        (assoc-in [:config :topics] topic-name)
        (assoc-in [:config :topic.index.map] (str (clojure.string/upper-case index-name) ":" (clojure.string/lower-case index-name)))

        )))


(defmethod p/gen-connector
  "org.apache.kafka.connect.file.FileStreamSinkConnector"
  [connector-schema]
  ;(println schema)
  (let [f-name (get connector-schema :entity_name)
        ref-name (get connector-schema :ref_name)
        ref-name (if (or (nil? ref-name)
                         (clojure.string/blank? ref-name))
                   nil
                   ref-name)
        topic-name (clojure.string/lower-case f-name)

        ref-name (if-let [w ref-name]
                   (str f-name "_" w)
                   f-name)

        ;index-name (or ref-name topic-name)
        template-v (get connector-schema :template )
        template-name (get connector-schema :template-file-name)
        file-path (str (get-in template-v [:config :file] )  topic-name)
        ]

    (-> template-v
        (assoc :name (clojure.string/lower-case (str ref-name "_" template-name)))
        (assoc-in [:config :topics] topic-name)
        (assoc-in [:config :file] file-path)

        )))

