(ns ksqldb.client.connector-api
  (:require [cheshire.core :as json]
            [clj-http.client :as http]
            [ksqldb.client.protocol :as c]
            [clojure.tools.logging :as log]))


; :connector-url  "http://localhost:8083/connectors"
(defn get-connector-url [client]
  (str (get client :connector-url) "/connectors"))

;;;;;;;;;;;;;;;;;;;;;;;;;;Connector

(defmethod c/invoke "create-connector"
  [client {:keys [request]}]
  (let [w (json/generate-string request {:pretty true})]
    (http/post (get-connector-url client)
               {:body    w
                :headers {"X-Api-Version" "2"
                          "Content-Type"  "application/json"}
                :accept  :json})))


(defmethod c/invoke "create-connector-batch"
  [client {:keys [request]}]
  (try (doseq [m request]
         (log/info "creating new connector " (:name m))
         (c/invoke client {:op "create-connector" :request m}))
       (catch Exception e
         (do
           (println "--" (ex-data e))
           (log/warn "connector fails " (-> (ex-data e)
                                            :body
                                            (json/parse-string true)))
           )
         )))

(defmethod c/invoke "show-connectors"
  [client {:keys [request]}]
  (->> (http/get (get-connector-url client))
       :body
       (json/parse-string)
       (into #{})))

(defmethod c/invoke "describe-connector"
  [client {:keys [request]}]
  (-> (http/get (str (get-connector-url client) "/" request))
      (:body)
      (json/parse-string true)))


(defmethod c/invoke "show-connectors-status"
  [client {:keys [request]}]
  (-> (http/get (str (get-connector-url client) "?expand=status"))
      :body
      (json/parse-string true)
      vals
      (->> (mapv vals))
      (flatten)))


(defmethod c/invoke "delete-connector"
  [client {:keys [request]}]
  (http/delete (str (get-connector-url client) "/" request)))



(defmethod c/invoke "pause-connector"
  [client {:keys [request]}]
  (-> (http/put (str (get-connector-url client) "/" request "/pause"))
      (:body)))

(defmethod c/invoke "resume-connector"
  [client {:keys [request]}]
  (-> (http/put (str (get-connector-url client) "/" request "/resume"))
      (:body)))



(defmethod c/invoke "delete-all-connector"
  [client {:keys [request]}]
  (let [coll (c/invoke client {:op "show-connectors-status"})]
    (doseq [c (into [] coll)]
      (http/delete (str (get-connector-url client) "/" (get c :name))))))


(defmethod c/invoke "pause-all-source-connector"
  [client {:keys [request]}]
  (let [connector-list (c/invoke client {:op "show-connectors-status"})
        source-connector-list (filter #(= (get % :type) "source") connector-list)]
    (doseq [c source-connector-list]
      (try
        (c/invoke client {:op "pause-connector" :request (get c :name)})
        (catch Exception e)))))


(defmethod c/invoke "resume-all-source-connector"
  [client {:keys [request]}]
  (let [connector-list (c/invoke client {:op "show-connectors-status"})
        source-connector-list (filter #(= (get % :type) "source") connector-list)]
    (doseq [c source-connector-list]
      (c/invoke client {:op "resume-connector" :request (get c :name)}))))


(defmethod c/invoke "pause-all-sink-connector"
  [client {:keys [request]}]
  (let [connector-list (c/invoke client {:op "show-connectors-status"})
        source-connector-list (filter #(= (get % :type) "sink") connector-list)]
    (doseq [c source-connector-list]
      (try
        (c/invoke client {:op "pause-connector" :request (get c :name)})
        (catch Exception e)))))


(defmethod c/invoke "resume-all-sink-connector"
  [client {:keys [request]}]
  (let [connector-list (c/invoke client {:op "show-connectors-status"})
        source-connector-list (filter #(= (get % :type) "sink") connector-list)]
    (doseq [c source-connector-list]
      (c/invoke client {:op "resume-connector" :request (get c :name)}))))