(ns kotr.client.stardog
  (:require [clojure.tools.logging :as log]
            [stardog.core :as stardog]
            [ksqldb.client :as client]
            [clojure.core.async :as async]
            [cheshire.core :as json]
            [ksqldb.client.broker-api :as repl]
            [ksqldb.client.broker-sedes-config :as cb]
            [kotr.client.stardog.kafka-consumer :as kc]
            [kotr.client.stardog.turtle-builder :as tb]))


(defmulti warp-request (fn [handler t] t))

(defmethod warp-request
  :avro
  [handler t]
  (fn [request]
    (handler request)))


(defmethod warp-request
  :json
  [handler t]
  (fn [request]
    (-> request
        (update :value json/parse-string)
        (handler))))


(defn warp-trutle [handler schema context]
  (fn [request]
    (let []
      (handler (tb/as-turtle-from-entity schema request)))))



(defn push-to-stardog [db-spec kafka-topic-name turtle-coll]
  (log/info (str " from topic  " kafka-topic-name " to stardog total:  ") (count turtle-coll))
  (with-open [c (stardog/connect db-spec)]
    (log/debug "inserting to stardog")
    (stardog/with-transaction [c]
                              (doseq [turtle turtle-coll]
                                (log/debug "push turtle " turtle)
                                ;   (log/info "--insert " turtle)
                                (stardog/insert! c turtle)))))



(defn get-consumer-config [context]
  {"bootstrap.servers"  (get context :kafka-broker)
   "auto.offset.reset"  "earliest"
   "enable.auto.commit" "false"
   "group.id"           (get context :group-id (str (java.util.UUID/randomUUID)))})



(defn start-consume-data [context handler topic-name]
  (let [schema (kc/get-schema context topic-name)
        t (if schema :avro :json)
        warp-handler (-> handler
                         (warp-trutle (or schema
                                          (kc/get-default-schema context topic-name)) context)
                         (warp-request t))
        topic-config (kc/get-topic-config context topic-name t)]
    (repl/consume2 (get-consumer-config context) topic-config warp-handler)))



(defn create-stardog-conn [{:keys [db url user pass reasoning class-uri instance-uri]}]
  (let [db-spec (stardog/create-db-spec db url user pass reasoning)
        [v uri] class-uri
        [vv urii] instance-uri]
    (alter-var-root #'tb/class-uri (constantly v))
    (alter-var-root #'tb/instance-uri (constantly vv))
    (with-open [c (stardog/connect db-spec)]
      (stardog/add-ns! c v uri)
      (stardog/add-ns! c vv urii))
    db-spec))


(defn topic-collector [context topic-name-ch]
  (log/debug "start collect topic with " topic-name-ch)
  (async/go-loop []
    (doseq [v (kc/get-topic-list context)]
      (async/>! topic-name-ch v))
    (async/timeout 200000)
    (recur)))


(def topic-sets (atom #{}))
(def black-list (atom {}))


(defn run-process [file-name]
  (let [context (clojure.edn/read-string (slurp file-name))
        t-channel (async/chan 1)
        stardog-config (get context :stardog)
        db-spec (create-stardog-conn stardog-config)]

    (log/info "staring service with " context)
    (topic-collector context t-channel)

    (loop [out []]
      (let [kafa-topic-name (async/<!! t-channel)]
        ; (log/debug "Got new topic " kafa-topic-name)
        (when (and (not (contains? @topic-sets kafa-topic-name))
                   (< (get @black-list kafa-topic-name 0) 6))
          (log/info "start processing new topic " kafa-topic-name)
          (swap! topic-sets conj kafa-topic-name)
          (async/go
            (try

              (start-consume-data context (partial push-to-stardog db-spec kafa-topic-name) kafa-topic-name)

              (catch Exception e
                (do
                  (swap! topic-sets disj kafa-topic-name)
                  (log/error "Error for topic " e)
                  (swap! black-list (fn [v]
                                      (if-let [t-name (get v kafa-topic-name)]
                                        (update v kafa-topic-name inc)
                                        (assoc v kafa-topic-name 0)))))))))
        (recur out)))))



(defn -main [& args]
  (let [[file-name] args]
    (log/info "staring stardog kafka bridge with file " file-name)
    (run-process file-name)
    #_(run-process-by-file-name file-name)))


(comment

  (swap! client/consume-state-atom (fn [m] {"_pipeline_metadata1_" true}))

  @client/consume-state-atom

  (reset! client/consume-state-atom {})

  (run-process-by-file-name "lconfig.edn")


  (run-process "lconfig.edn")





  (start-consume-data (clojure.edn/read-string (slurp "lconfig.edn"))
                      println
                      "user_raw"

                      )

  ;(clojure.walk/keywordize-keys {"fname":"lname","lname":"test","id":"1","age":"20"})


  (start-consume-data (clojure.edn/read-string (slurp "lconfig.edn"))
                      println
                      "user_refined"

                      )



  (kc/get-topic-config (clojure.edn/read-string (slurp "lconfig.edn")) "user_refined")






  (repl/get (clojure.edn/read-string (slurp "lconfig.edn")))


  (repl/topics (clojure.edn/read-string (slurp "lconfig.edn")))


  (let [w (clojure.edn/read-string (slurp "lconfig.edn"))
        data-chennel (async/chan 1)
        ]

    )



  (-> (clojure.edn/read-string (slurp "lconfig.edn"))
      (repl/consume (cb/edn-topic-config "_pipeline_metadata") (fn [w]
                                                                 (println "-" 2)
                                                                 ))

      )




  (let [c (clojure.edn/read-string (slurp "lconfig.edn"))]
    ; (cb/avro-topic-config c "ref_mapping")
    (-> c
        (repl/consume (cb/avro-topic-config c "tnf_party") (fn [w]
                                                             (println "-" w)
                                                             ))

        )
    )


  (ksql/topic "ID3_PERSON_IN" 1 println)


  (kc/add-metadata-listener "_pipeline_metadata" (fn [k v]
                                                   (println "---value " v)))

  ;(boolean "true")






  (let [test-db-spec (stardog/create-db-spec "tnf_db" "http://localhost:5820/" "admin" "admin" true)]
    (with-open [c (stardog/connect test-db-spec)]
      (stardog/list-namespaces c))

    )



  (let [schema {:name   "person"
                :key    "id"
                :fields [{:name "id" :type "int"}
                         {:name "fnmae" :type "string"}]}
        v [{"id" 1 :fname "test" :lname "fname"}
           {"id" 2 :fname "test" :lname "fname"}]]
    (->> (avro-to-turtle schema v)
         (push-to-stardog test-db-spec))
    )


  (with-open [c (get-connection)]
    (star/add-ns! c "allianz" "http://ontologies.allianz.com/core#"))


  (with-open [c (get-connection)]
    (star/with-transaction [c]
                           (star/insert! c ["urn:a:product-1" "urn:a:type" "allianz:Product"])
                           ;(star/insert! c ["allianz:a:product-1" "rdf:a:type" "allianz:Product"])
                           ))

  (let [c (get-connection)]
    (star/query c "select ?n { .... }" {:converter str})
    )

  )
