(ns {{sanitized-ns}}.server.config
  (:require [clojure.java.io :as io]))

;; This is brought to you by Craig Andera (see https://gist.github.com/candera/4565367)


(defn form-seq
  "Lazy seq of forms read from a reader"
  [reader]
  (let [form (read reader false reader)]
    (when-not (= form reader)
      (cons form (lazy-seq (form-seq reader))))))

(defmulti invoke-extension
  "An extension point for operations in config files."
  (fn [_ operation & _] operation))

(defn mapify
  "Turns a form read from a config file at `origin` into a map."
  [form origin]
  (cond
    (map? form) form
    (list? form) (let [[operation & args] form]
                   (invoke-extension origin operation args))
    :else (throw (ex-info (str "No support for mapifying form " (pr-str form))
                   {:reason :cant-mapify
                    :form form
                    :origin origin}))))

(defn read-config
  "Returns a map read from `path`. Map will be generated by merging all
  forms found in file. Lists are interpreted as invocations of the
  `invoke-extension` multimethod, dispatched on the symbol in the
  first position.
 
  Example:
 
  (include \"something.edn\")
  ;; Comments are ignored
  {:foo :bar
   :bar 1234}
 
  Might yield:
 
  {:foo :bar
   :from-something 4321
   :bar 1234}"
  [path]
  (let [source (-> path
                 io/file
                 java.io.FileReader.
                 java.io.PushbackReader.)]
    (reduce merge (map #(mapify % path) (form-seq source)))))

(defn path-relative-to
  "Given two paths `p1` and `p2`, returns a path that is the
  combination of them. If `p2` is absolute, `p2` is returned.
  Otherwise p1/p2 is returned."
  [p1 p2]
  (let [f1 (java.io.File. p1)
        f2 (java.io.File. p2)]
    (if (.isAbsolute f2)
      p2
      (.getPath (java.io.File. p1 p2)))))

(defn parent
  "Returns the parent directory of `path`"
  [path]
  (.getParent (io/file path)))

;; E.g. (include "foo.edn") => {:some :map}
;; TODO: globbing support
(defmethod invoke-extension 'include
  [origin operation [path]]
  (read-config (path-relative-to (parent origin) path)))


(defn init []
  (defonce config (ref (read-config (io/resource "config.edn")))))