;; Copyright (C) 2011, Eduardo Julián. All rights reserved.
;;
;; The use and distribution terms for this software are covered by the 
;; Eclipse Public License 1.0
;; (http://opensource.org/licenses/eclipse-1.0.php) which can be found
;; in the file epl-v10.html at the root of this distribution.
;;
;; By using this software in any fashion, you are agreeing to be bound
;; by the terms of this license.
;;
;; You must not remove this notice, or any other, from this software.

(ns clj-yelp.v1
  "This namespace gives access to the v1.0 APIs: Review Search API, Phone API & Neighborhood API."
  (:import (java.net URL URLEncoder))
  (:use (clojure.contrib def))
  (:require [clj-http.client :as client]
    (clojure.contrib [json :as json])))

; Util fns
(defn- url-encode [url] (URLEncoder/encode url "UTF-8"))

(defn- fetch-url [url] (-> url client/get :body json/read-json))

(defn- append-arg
  ([url] url)
  ([url argk argv] (if argv (str url "&" argk "=" argv) url))
  ([url argk argv & args] (apply append-arg (append-arg url argk argv) args)))

(defn- extract-vars [pairs] (->> (map second (partition 2 pairs)) (map #(if (list? %) (second %) %))))

(def #^{:doc "The Yelp Web Service ID."} *ywsid*)

(defmacro with-ywsid
  "Binds *ywsid* to the given ID to be used by v1.0 API calls."
  [ywsid & forms] `(binding [*ywsid* ~ywsid] ~@forms))

(defmacro- json-try [sexp]
  `(let [res# ~sexp]
     (if-not (-> res# :message :text (= "OK"))
       (throw (Exception. (str "Code " (-> res# :message :code) ": " (-> res# :message :text))))
       res#)))

(defmacro- make-fns [& specifics]
  `(do ~@(for [[fn-sym fn-url doc-str appends] (partition 4 specifics)]
           `(defn ~fn-sym ~doc-str "" [{:keys ~(vec (extract-vars appends))}]
              (-> ~fn-url (str "?ywsid=" *ywsid*) (append-arg ~@appends) fetch-url json-try)
              ))))

(make-fns
  ; Review Search API
  search-business-reviews "http://api.yelp.com/business_review_search" ""
  ["term" term, "limit" limit,
   "tl_lat" tl-lat, "tl_long" tl-lng, "br_lat" br-lat, "br_long" br-lng,
   "lat" lat, "long" lng, "radius" radius,
   "location" location, "cc" country-code, "category" category]
  ; Phone API
  find-by-phone "http://api.yelp.com/phone_search" ""
  ["phone" phone, "cc" country-code]
  ; Neighborhood API
  find-neighborhood "http://api.yelp.com/neighborhood_search" ""
  ["lat" lat, "long" lng, "location" location, "cc" country-code]
  )
