;; Copyright (C) 2011, Eduardo Julián. All rights reserved.
;;
;; The use and distribution terms for this software are covered by the 
;; Eclipse Public License 1.0
;; (http://opensource.org/licenses/eclipse-1.0.php) which can be found
;; in the file epl-v10.html at the root of this distribution.
;;
;; By using this software in any fashion, you are agreeing to be bound
;; by the terms of this license.
;;
;; You must not remove this notice, or any other, from this software.

(ns clj-yelp.v2
  "This namespace gives access to the v2.0 APIs: Search API, Business API."
  (:import (org.scribe.builder.api DefaultApi10a)
    (org.scribe.builder ServiceBuilder)
    (org.scribe.model Token Verb OAuthRequest)
    (org.scribe.oauth OAuthService))
  (:use (clojure.contrib def))
  (:require (clojure.contrib [json :as json])))

(def YelpAPI2
  (class
    (proxy [org.scribe.builder.api.DefaultApi10a] []
      (getAccessTokenEndpoint [] nil)
      (getAuthorizationUrl [token] nil)
      (getRequestTokenEndpoint [] nil))))

(def #^{:doc "Holds the current OAuth Service."} *oauth-service*)
(def #^{:doc "Holds the current OAuth Access Token."} *access-token*)

(defn make-service
  "Given the consumer key and secrets, builds an OAuth Service."
  [consumer-key, consumer-secret]
  (-> (ServiceBuilder.) (.provider YelpAPI2) (.apiKey consumer-key) (.apiSecret consumer-secret) .build))

(defn make-access-token
  "Given a request token and token secret, creates an access token."
  [token, token-secret]
  (Token. token token-secret))

(defmacro with-oauth
"Binds up the *oauth-service* and *access-token* vars to be used by v2.0 fns."
  [oauth & forms]
  `(binding [*oauth-service* (make-service (:consumer-key ~oauth) (:consumer-secret ~oauth)),
             *access-token* (make-access-token (:token ~oauth) (:token-secret ~oauth))]
     ~@forms))

(defmacro- add-param [request key val]
  `(when ~val (.addQuerystringParameter ~request ~key ~val)))

(defmacro- json-try [sexp]
  `(let [res# ~sexp]
     (if (:error res#)
       (throw (Exception. (str (-> res# :error :id) ": " (-> res# :error :text))))
       res#)))

; Search API
(defn search-businesses
"Given the desired optional values, searches for businesses.
Take into account that:
coordinates (provided as a [vector]) == ll
current-coordinates (provided as a [vector]) == cll

The rest of the params are pretty obvious and require no explanation."
  [{:keys [term limit results-offset sort-mode category
           radius country-code language bounds location
           coordinates current-coordinates]}]
  (let [request (OAuthRequest. Verb/GET "http://api.yelp.com/v2/search")]
    (do (add-param request "term" term) (add-param request "sort" sort-mode)
      (add-param request "limit" limit) (add-param request "offset" results-offset)
      (add-param request "category_filter" category) (add-param request "radius_filter" radius)
      (add-param request "cc" country-code) (add-param request "lang" language)
      (add-param request "bounds" (when bounds (str (first bounds) "," (second bounds) "|" (nth bounds 2) "," (nth bounds 3))))
      (add-param request "location" location)
      (add-param request "ll" (when coordinates (apply str (butlast (interleave coordinates (repeat ","))))))
      (add-param request "cll" (when current-coordinates (str (first current-coordinates) "," (second current-coordinates)))))
    (.signRequest *oauth-service* *access-token* request)
    (-> request .send .getBody json/read-json json-try)))

; Business API
(defn lookup-business
  "Given the business-id as a \"string\", looks it up and returns the results."
  [business-id]
  (let [request (OAuthRequest. Verb/GET (str "http://api.yelp.com/v2/business/" business-id))]
    (.signRequest *oauth-service* *access-token* request)
    (-> request .send .getBody json/read-json json-try)))
