(ns to-utf8-unix.middlewares
  "It contains a set of middlewares for converting file content.
  Each middleware can be defined as below.
  (fn [next_middleware]
    (fn [buffer_descriptor]
      (do-something)
      (next_middleware modified_buffer_descriptor)))

  A middleware may choose not to call the next one but to return
  modified buffer_descriptor directly to stop middleware processing.

  buffer_descriptor is defined in README.md"
  (:require [clojure.string :as string]))

(def charsetDetector (js/require "node-icu-charset-detector"))
(def iconv (js/require "iconv-lite"))
(def cr (js/require "cr"))

(defn detect_charset
  [next]
  (fn [{:keys [buffer] :as buf_desc}]
    (let [charset (.detectCharset charsetDetector buffer)]
      (next (assoc buf_desc
                   :charset (.toString charset)
                   :confidence (.-confidence charset))))))

(defn ensure_confidence
  [confidence_threshold]
  (fn [next]
    (fn [{:keys [confidence] :as buf_desc}]
      (if (< confidence confidence_threshold)
        (assoc buf_desc
               :message (str "Charset detection confidence is below "
                             confidence_threshold)
               :save false)
        (next buf_desc)))))

(defn encode_charset
  [target_charset]
  (fn [next]
    (fn [{:keys [charset buffer] :as buf_desc}]
      (let [buf_desc (if (= (type buffer) js/String)
                       buf_desc
                       (if (= charset target_charset)
                         (assoc buf_desc
                                :buffer (.toString buffer target_charset))
                         (assoc buf_desc
                                :buffer (.decode iconv buffer charset)
                                :save true)))]
        (next buf_desc)))))

(defn remove_carriage_return
  "buffer should be js/String."
  [next]
  (fn [{:keys [buffer] :as buf_desc}]
    (let [buf_desc (if (string/index-of buffer "\r")
                     (assoc buf_desc :buffer (cr buffer) :save true)
                     buf_desc)]
      (next buf_desc))))

(defn add_conversion_msg
  [next]
  (fn [{:keys [save] :as buf_desc}]
    (next (assoc buf_desc :message (if save
                                     "Conversion done."
                                     "Conversion not needed.")))))
