(ns org.apache.commons.math4.linear.BlockRealMatrix
  "Cache-friendly implementation of RealMatrix using a flat arrays to store
  square blocks of the matrix.

  This implementation is specially designed to be cache-friendly. Square blocks are
  stored as small arrays and allow efficient traversal of data both in row major direction
  and columns major direction, one block at a time. This greatly increases performances
  for algorithms that use crossed directions loops like multiplication or transposition.


  The size of square blocks is a static parameter. It may be tuned according to the cache
  size of the target computer processor. As a rule of thumbs, it should be the largest
  value that allows three blocks to be simultaneously cached (this is necessary for example
  for matrix multiplication). The default value is to use 52x52 blocks which is well suited
  for processors with 64k L1 cache (one block holds 2704 values or 21632 bytes). This value
  could be lowered to 36x36 for processors with 32k L1 cache.


  The regular blocks represent BLOCK_SIZE x BLOCK_SIZE squares. Blocks
  at right hand side and bottom side which may be smaller to fit matrix dimensions. The square
  blocks are flattened in row major order in single dimension arrays which are therefore
  BLOCK_SIZE2 elements long for regular blocks. The blocks are themselves
  organized in row major order.


  As an example, for a block size of 52x52, a 100x60 matrix would be stored in 4 blocks.
  Block 0 would be a double[2704] array holding the upper left 52x52 square, block 1 would be
  a double[416] array holding the upper right 52x8 rectangle, block 2 would be a double[2496]
  array holding the lower left 48x52 rectangle and block 3 would be a double[384] array
  holding the lower right 48x8 rectangle.


  The layout complexity overhead versus simple mapping of matrices to java
  arrays is negligible for small matrices (about 1%). The gain from cache efficiency leads
  to up to 3-fold improvements for matrices of moderate to large size."
  (:refer-clojure :only [require comment defn ->])
  (:import [org.apache.commons.math4.linear BlockRealMatrix]))

(defn ->block-real-matrix
  "Constructor.

  Create a new dense matrix copying entries from block layout data.
   The input array must already be in blocks layout.

  rows - Number of rows in the new matrix. - `int`
  columns - Number of columns in the new matrix. - `int`
  block-data - data for new matrix - `double[][]`
  copy-array - Whether the input array will be copied or referenced. - `boolean`

  throws: org.apache.commons.math4.exception.DimensionMismatchException - if the shape of blockData is inconsistent with block layout."
  (^BlockRealMatrix [^Integer rows ^Integer columns block-data ^Boolean copy-array]
    (new BlockRealMatrix rows columns block-data copy-array))
  (^BlockRealMatrix [^Integer rows ^Integer columns]
    (new BlockRealMatrix rows columns))
  (^BlockRealMatrix [raw-data]
    (new BlockRealMatrix raw-data)))

(def *-block-size
  "Static Constant.

  Block size.

  type: int"
  BlockRealMatrix/BLOCK_SIZE)

(defn *to-blocks-layout
  "Convert a data array from raw layout to blocks layout.

   Raw layout is the straightforward layout where element at row i and
   column j is in array element rawData[i][j]. Blocks layout
   is the layout used in BlockRealMatrix instances, where the matrix
   is split in square blocks (except at right and bottom side where blocks may
   be rectangular to fit matrix size) and each block is stored in a flattened
   one-dimensional array.


   This method creates an array in blocks layout from an input array in raw layout.
   It can be used to provide the array argument of the BlockRealMatrix(int, int, double[][], boolean) constructor.

  raw-data - Data array in raw layout. - `double[][]`

  returns: a new data array containing the same entries but in blocks layout. - `double[][]`

  throws: org.apache.commons.math4.exception.DimensionMismatchException - if rawData is not rectangular."
  ([raw-data]
    (BlockRealMatrix/toBlocksLayout raw-data)))

(defn *create-blocks-layout
  "Create a data array in blocks layout.

   This method can be used to create the array argument of the BlockRealMatrix(int, int, double[][], boolean) constructor.

  rows - Number of rows in the new matrix. - `int`
  columns - Number of columns in the new matrix. - `int`

  returns: a new data array in blocks layout. - `double[][]`"
  ([^Integer rows ^Integer columns]
    (BlockRealMatrix/createBlocksLayout rows columns)))

(defn scalar-add
  "Returns the result of adding d to each entry of this.

  d - value to be added to each entry - `double`

  returns: d  this - `org.apache.commons.math4.linear.BlockRealMatrix`"
  (^org.apache.commons.math4.linear.BlockRealMatrix [^BlockRealMatrix this ^Double d]
    (-> this (.scalarAdd d))))

(defn pre-multiply
  "Returns the (row) vector result of premultiplying this by the vector v.

  v - the row vector to premultiply by - `double[]`

  returns: v * this - `double[]`

  throws: org.apache.commons.math4.exception.DimensionMismatchException - if the length of v does not match the row dimension of this."
  ([^BlockRealMatrix this v]
    (-> this (.preMultiply v))))

(defn set-row-vector
  "Sets the specified row of this matrix to the entries of
   the specified vector. Row indices start at 0.

  row - Row to be set. - `int`
  vector - row vector to be copied (must have the same number of column as the instance). - `org.apache.commons.math4.linear.RealVector`

  throws: org.apache.commons.math4.exception.OutOfRangeException - if the specified row index is invalid."
  ([^BlockRealMatrix this ^Integer row ^org.apache.commons.math4.linear.RealVector vector]
    (-> this (.setRowVector row vector))))

(defn create-matrix
  "Create a new RealMatrix of the same type as the instance with the
   supplied
   row and column dimensions.

  row-dimension - the number of rows in the new matrix - `int`
  column-dimension - the number of columns in the new matrix - `int`

  returns: a new matrix of the same type as the instance - `org.apache.commons.math4.linear.BlockRealMatrix`

  throws: org.apache.commons.math4.exception.NotStrictlyPositiveException - if row or column dimension is not positive."
  (^org.apache.commons.math4.linear.BlockRealMatrix [^BlockRealMatrix this ^Integer row-dimension ^Integer column-dimension]
    (-> this (.createMatrix row-dimension column-dimension))))

(defn get-data
  "Returns matrix entries as a two-dimensional array.

  returns: 2-dimensional array of entries - `double[][]`"
  ([^BlockRealMatrix this]
    (-> this (.getData))))

(defn get-column-matrix
  "Get the entries at the given column index as a column matrix. Column
   indices start at 0.

  column - Column to be fetched. - `int`

  returns: column Matrix. - `org.apache.commons.math4.linear.BlockRealMatrix`

  throws: org.apache.commons.math4.exception.OutOfRangeException - if the specified column index is invalid."
  (^org.apache.commons.math4.linear.BlockRealMatrix [^BlockRealMatrix this ^Integer column]
    (-> this (.getColumnMatrix column))))

(defn get-column-dimension
  "Returns the number of columns of this matrix.

  returns: the number of columns. - `int`"
  (^Integer [^BlockRealMatrix this]
    (-> this (.getColumnDimension))))

(defn copy
  "Returns a (deep) copy of this.

  returns: matrix copy - `org.apache.commons.math4.linear.BlockRealMatrix`"
  (^org.apache.commons.math4.linear.BlockRealMatrix [^BlockRealMatrix this]
    (-> this (.copy))))

(defn scalar-multiply
  "Returns the result of multiplying each entry of this by
   d.

  d - value to multiply all entries by - `double`

  returns: d * this - `org.apache.commons.math4.linear.RealMatrix`"
  (^org.apache.commons.math4.linear.RealMatrix [^BlockRealMatrix this ^Double d]
    (-> this (.scalarMultiply d))))

(defn multiply
  "Returns the result of postmultiplying this by m.

  m - matrix to postmultiply by - `org.apache.commons.math4.linear.RealMatrix`

  returns: this * m - `org.apache.commons.math4.linear.BlockRealMatrix`

  throws: org.apache.commons.math4.exception.DimensionMismatchException - if columnDimension(this) != rowDimension(m)"
  (^org.apache.commons.math4.linear.BlockRealMatrix [^BlockRealMatrix this ^org.apache.commons.math4.linear.RealMatrix m]
    (-> this (.multiply m))))

(defn get-column
  "Get the entries at the given column index as an array. Column indices
   start at 0.

  column - Column to be fetched. - `int`

  returns: the array of entries in the column. - `double[]`

  throws: org.apache.commons.math4.exception.OutOfRangeException - if the specified column index is not valid."
  ([^BlockRealMatrix this ^Integer column]
    (-> this (.getColumn column))))

(defn get-entry
  "Get the entry in the specified row and column. Row and column indices
   start at 0.

  row - Row index of entry to be fetched. - `int`
  column - Column index of entry to be fetched. - `int`

  returns: the matrix entry at (row, column). - `double`

  throws: org.apache.commons.math4.exception.OutOfRangeException - if the row or column index is not valid."
  (^Double [^BlockRealMatrix this ^Integer row ^Integer column]
    (-> this (.getEntry row column))))

(defn operate
  "Returns the result of multiplying this by the vector v.

  v - the vector to operate on - `double[]`

  returns: this * v - `double[]`

  throws: org.apache.commons.math4.exception.DimensionMismatchException - if the length of v does not match the column dimension of this."
  ([^BlockRealMatrix this v]
    (-> this (.operate v))))

(defn walk-in-optimized-order
  "Visit (and possibly change) some matrix entries using the fastest possible order.
   The fastest walking order depends on the exact matrix class. It may be
   different from traditional row or column orders.

  visitor - visitor used to process all matrix entries - `org.apache.commons.math4.linear.RealMatrixChangingVisitor`
  start-row - Initial row index - `int`
  end-row - Final row index (inclusive) - `int`
  start-column - Initial column index - `int`
  end-column - Final column index (inclusive) - `int`

  returns: the value returned by RealMatrixChangingVisitor.end() at the end
   of the walk - `double`

  throws: org.apache.commons.math4.exception.OutOfRangeException - if the indices are not valid."
  (^Double [^BlockRealMatrix this ^org.apache.commons.math4.linear.RealMatrixChangingVisitor visitor ^Integer start-row ^Integer end-row ^Integer start-column ^Integer end-column]
    (-> this (.walkInOptimizedOrder visitor start-row end-row start-column end-column)))
  (^Double [^BlockRealMatrix this ^org.apache.commons.math4.linear.RealMatrixChangingVisitor visitor]
    (-> this (.walkInOptimizedOrder visitor))))

(defn get-column-vector
  "Get the entries at the given column index as a vector. Column indices
   start at 0.

  column - Column to be fetched. - `int`

  returns: a column vector. - `org.apache.commons.math4.linear.RealVector`

  throws: org.apache.commons.math4.exception.OutOfRangeException - if the specified column index is invalid"
  (^org.apache.commons.math4.linear.RealVector [^BlockRealMatrix this ^Integer column]
    (-> this (.getColumnVector column))))

(defn get-sub-matrix
  "Gets a submatrix. Rows and columns are indicated
   counting from 0 to n-1.

  start-row - Initial row index - `int`
  end-row - Final row index (inclusive) - `int`
  start-column - Initial column index - `int`
  end-column - Final column index (inclusive) - `int`

  returns: The subMatrix containing the data of the
   specified rows and columns. - `org.apache.commons.math4.linear.BlockRealMatrix`

  throws: org.apache.commons.math4.exception.OutOfRangeException - if the indices are not valid."
  (^org.apache.commons.math4.linear.BlockRealMatrix [^BlockRealMatrix this ^Integer start-row ^Integer end-row ^Integer start-column ^Integer end-column]
    (-> this (.getSubMatrix start-row end-row start-column end-column))))

(defn get-row-dimension
  "Returns the number of rows of this matrix.

  returns: the number of rows. - `int`"
  (^Integer [^BlockRealMatrix this]
    (-> this (.getRowDimension))))

(defn set-column
  "Sets the specified column of this matrix to the entries
   of the specified array. Column indices start at 0.

  column - Column to be set. - `int`
  array - Column array to be copied (must have the same number of rows as the instance). - `double[]`

  throws: org.apache.commons.math4.exception.OutOfRangeException - if the specified column index is invalid."
  ([^BlockRealMatrix this ^Integer column array]
    (-> this (.setColumn column array))))

(defn set-row
  "Sets the specified row of this matrix to the entries
   of the specified array. Row indices start at 0.

  row - Row to be set. - `int`
  array - Row matrix to be copied (must have the same number of columns as the instance) - `double[]`

  throws: org.apache.commons.math4.exception.OutOfRangeException - if the specified row index is invalid."
  ([^BlockRealMatrix this ^Integer row array]
    (-> this (.setRow row array))))

(defn set-row-matrix
  "Sets the specified row of this matrix to the entries of
   the specified row matrix. Row indices start at 0.

  row - Row to be set. - `int`
  matrix - Row matrix to be copied (must have one row and the same number of columns as the instance). - `org.apache.commons.math4.linear.RealMatrix`

  throws: org.apache.commons.math4.exception.OutOfRangeException - if the specified row index is invalid."
  ([^BlockRealMatrix this ^Integer row ^org.apache.commons.math4.linear.RealMatrix matrix]
    (-> this (.setRowMatrix row matrix))))

(defn subtract
  "Returns this minus m.

  m - matrix to be subtracted - `org.apache.commons.math4.linear.RealMatrix`

  returns: this - m - `org.apache.commons.math4.linear.BlockRealMatrix`

  throws: org.apache.commons.math4.linear.MatrixDimensionMismatchException - if m is not the same size as this."
  (^org.apache.commons.math4.linear.BlockRealMatrix [^BlockRealMatrix this ^org.apache.commons.math4.linear.RealMatrix m]
    (-> this (.subtract m))))

(defn set-column-matrix
  "Sets the specified column of this matrix to the entries
   of the specified column matrix. Column indices start at 0.

  column - Column to be set. - `int`
  matrix - Column matrix to be copied (must have one column and the same number of rows as the instance). - `org.apache.commons.math4.linear.RealMatrix`

  throws: org.apache.commons.math4.exception.OutOfRangeException - if the specified column index is invalid."
  ([^BlockRealMatrix this ^Integer column ^org.apache.commons.math4.linear.RealMatrix matrix]
    (-> this (.setColumnMatrix column matrix))))

(defn get-row
  "Get the entries at the given row index. Row indices start at 0.

  row - Row to be fetched. - `int`

  returns: the array of entries in the row. - `double[]`

  throws: org.apache.commons.math4.exception.OutOfRangeException - if the specified row index is not valid."
  ([^BlockRealMatrix this ^Integer row]
    (-> this (.getRow row))))

(defn get-row-vector
  "Returns the entries in row number row as a vector. Row indices
   start at 0.

  row - Row to be fetched. - `int`

  returns: a row vector. - `org.apache.commons.math4.linear.RealVector`

  throws: org.apache.commons.math4.exception.OutOfRangeException - if the specified row index is invalid."
  (^org.apache.commons.math4.linear.RealVector [^BlockRealMatrix this ^Integer row]
    (-> this (.getRowVector row))))

(defn get-norm
  "Returns the
   maximum absolute row sum norm of the matrix.

  returns: norm - `double`"
  (^Double [^BlockRealMatrix this]
    (-> this (.getNorm))))

(defn set-entry
  "Set the entry in the specified row and column. Row and column indices
   start at 0.

  row - Row index of entry to be set. - `int`
  column - Column index of entry to be set. - `int`
  value - the new value of the entry. - `double`

  throws: org.apache.commons.math4.exception.OutOfRangeException - if the row or column index is not valid"
  ([^BlockRealMatrix this ^Integer row ^Integer column ^Double value]
    (-> this (.setEntry row column value))))

(defn add-to-entry
  "Adds (in place) the specified value to the specified entry of
   this matrix. Row and column indices start at 0.

  row - Row index of the entry to be modified. - `int`
  column - Column index of the entry to be modified. - `int`
  increment - value to add to the matrix entry. - `double`

  throws: org.apache.commons.math4.exception.OutOfRangeException - if the row or column index is not valid."
  ([^BlockRealMatrix this ^Integer row ^Integer column ^Double increment]
    (-> this (.addToEntry row column increment))))

(defn multiply-entry
  "Multiplies (in place) the specified entry of this matrix by the
   specified value. Row and column indices start at 0.

  row - Row index of the entry to be modified. - `int`
  column - Column index of the entry to be modified. - `int`
  factor - Multiplication factor for the matrix entry. - `double`

  throws: org.apache.commons.math4.exception.OutOfRangeException - if the row or column index is not valid."
  ([^BlockRealMatrix this ^Integer row ^Integer column ^Double factor]
    (-> this (.multiplyEntry row column factor))))

(defn add
  "Returns the sum of this and m.

  m - matrix to be added - `org.apache.commons.math4.linear.RealMatrix`

  returns: this  m - `org.apache.commons.math4.linear.BlockRealMatrix`

  throws: org.apache.commons.math4.linear.MatrixDimensionMismatchException - if m is not the same size as this."
  (^org.apache.commons.math4.linear.BlockRealMatrix [^BlockRealMatrix this ^org.apache.commons.math4.linear.RealMatrix m]
    (-> this (.add m))))

(defn get-frobenius-norm
  "Returns the
   Frobenius norm of the matrix.

  returns: norm - `double`"
  (^Double [^BlockRealMatrix this]
    (-> this (.getFrobeniusNorm))))

(defn get-row-matrix
  "Get the entries at the given row index as a row matrix.  Row indices start
   at 0.

  row - Row to be fetched. - `int`

  returns: row Matrix. - `org.apache.commons.math4.linear.BlockRealMatrix`

  throws: org.apache.commons.math4.exception.OutOfRangeException - if the specified row index is invalid."
  (^org.apache.commons.math4.linear.BlockRealMatrix [^BlockRealMatrix this ^Integer row]
    (-> this (.getRowMatrix row))))

(defn walk-in-row-order
  "Visit (and possibly change) some matrix entries in row order.
   Row order starts at upper left and iterating through all elements
   of a row from left to right before going to the leftmost element
   of the next row.

  visitor - visitor used to process all matrix entries - `org.apache.commons.math4.linear.RealMatrixChangingVisitor`
  start-row - Initial row index - `int`
  end-row - Final row index (inclusive) - `int`
  start-column - Initial column index - `int`
  end-column - Final column index - `int`

  returns: the value returned by RealMatrixChangingVisitor.end() at the end
   of the walk - `double`

  throws: org.apache.commons.math4.exception.OutOfRangeException - if the indices are not valid."
  (^Double [^BlockRealMatrix this ^org.apache.commons.math4.linear.RealMatrixChangingVisitor visitor ^Integer start-row ^Integer end-row ^Integer start-column ^Integer end-column]
    (-> this (.walkInRowOrder visitor start-row end-row start-column end-column)))
  (^Double [^BlockRealMatrix this ^org.apache.commons.math4.linear.RealMatrixChangingVisitor visitor]
    (-> this (.walkInRowOrder visitor))))

(defn transpose
  "Returns the transpose of this matrix.

  returns: transpose matrix - `org.apache.commons.math4.linear.BlockRealMatrix`"
  (^org.apache.commons.math4.linear.BlockRealMatrix [^BlockRealMatrix this]
    (-> this (.transpose))))

(defn set-sub-matrix
  "Replace the submatrix starting at row, column using data in the
   input subMatrix array. Indexes are 0-based.

   Example:
   Starting with

   1  2  3  4
   5  6  7  8
   9  0  1  2
   and subMatrix = {{3, 4} {5,6}}, invoking
   setSubMatrix(subMatrix,1,1)) will result in

   1  2  3  4
   5  3  4  8
   9  5  6  2

  sub-matrix - array containing the submatrix replacement data - `double[][]`
  row - row coordinate of the top, left element to be replaced - `int`
  column - column coordinate of the top, left element to be replaced - `int`

  throws: org.apache.commons.math4.exception.OutOfRangeException - if subMatrix does not fit into this matrix from element in (row, column)."
  ([^BlockRealMatrix this sub-matrix ^Integer row ^Integer column]
    (-> this (.setSubMatrix sub-matrix row column))))

(defn set-column-vector
  "Sets the specified column of this matrix to the entries
   of the specified vector. Column indices start at 0.

  column - Column to be set. - `int`
  vector - column vector to be copied (must have the same number of rows as the instance). - `org.apache.commons.math4.linear.RealVector`

  throws: org.apache.commons.math4.exception.OutOfRangeException - if the specified column index is invalid."
  ([^BlockRealMatrix this ^Integer column ^org.apache.commons.math4.linear.RealVector vector]
    (-> this (.setColumnVector column vector))))

