(ns javax.crypto.Cipher
  "This class provides the functionality of a cryptographic cipher for
  encryption and decryption. It forms the core of the Java Cryptographic
  Extension (JCE) framework.

  In order to create a Cipher object, the application calls the
  Cipher's getInstance method, and passes the name of the
  requested transformation to it. Optionally, the name of a provider
  may be specified.

  A transformation is a string that describes the operation (or
  set of operations) to be performed on the given input, to produce some
  output. A transformation always includes the name of a cryptographic
  algorithm (e.g., AES), and may be followed by a feedback mode and
  padding scheme.

   A transformation is of the form:


  `algorithm/mode/padding` or

  `algorithm`


   (in the latter case,
  provider-specific default values for the mode and padding scheme are used).
  For example, the following is a valid transformation:



      Cipher c = Cipher.getInstance(`AES/CBC/PKCS5Padding`);

  Using modes such as CFB and OFB, block
  ciphers can encrypt data in units smaller than the cipher's actual
  block size.  When requesting such a mode, you may optionally specify
  the number of bits to be processed at a time by appending this number
  to the mode name as shown in the `AES/CFB8/NoPadding` and
  `AES/OFB32/PKCS5Padding` transformations. If no such
  number is specified, a provider-specific default is used.
  Thus, block ciphers can be turned into byte-oriented stream ciphers by
  using an 8 bit mode such as CFB8 or OFB8.

  Modes such as Authenticated Encryption with Associated Data (AEAD)
  provide authenticity assurances for both confidential data and
  Additional Associated Data (AAD) that is not encrypted.  (Please see
   RFC 5116  for more
  information on AEAD and AEAD algorithms such as GCM/CCM.) Both
  confidential and AAD data can be used when calculating the
  authentication tag (similar to a Mac).  This tag is appended
  to the ciphertext during encryption, and is verified on decryption.

  AEAD modes such as GCM/CCM perform all AAD authenticity calculations
  before starting the ciphertext authenticity calculations.  To avoid
  implementations having to internally buffer ciphertext, all AAD data
  must be supplied to GCM/CCM implementations (via the updateAAD
  methods) before the ciphertext is processed (via
  the update and doFinal methods).

  Note that GCM mode has a uniqueness requirement on IVs used in
  encryption with a given key. When IVs are repeated for GCM
  encryption, such usages are subject to forgery attacks. Thus, after
  each encryption operation using GCM mode, callers should re-initialize
  the cipher objects with GCM parameters which has a different IV value.


      GCMParameterSpec s = ...;
      cipher.init(..., s);

      // If the GCM parameters were generated by the provider, it can
      // be retrieved by:
      // cipher.getParameters().getParameterSpec(GCMParameterSpec.class);

      cipher.updateAAD(...);  // AAD
      cipher.update(...);     // Multi-part update
      cipher.doFinal(...);    // conclusion of operation

      // Use a different IV value for every encryption
      byte[] newIv = ...;
      s = new GCMParameterSpec(s.getTLen(), newIv);
      cipher.init(..., s);
      ...
  Every implementation of the Java platform is required to support
  the following standard Cipher transformations with the keysizes
  in parentheses:

  AES/CBC/NoPadding (128)
  AES/CBC/PKCS5Padding (128)
  AES/ECB/NoPadding (128)
  AES/ECB/PKCS5Padding (128)
  DES/CBC/NoPadding (56)
  DES/CBC/PKCS5Padding (56)
  DES/ECB/NoPadding (56)
  DES/ECB/PKCS5Padding (56)
  DESede/CBC/NoPadding (168)
  DESede/CBC/PKCS5Padding (168)
  DESede/ECB/NoPadding (168)
  DESede/ECB/PKCS5Padding (168)
  RSA/ECB/PKCS1Padding (1024, 2048)
  RSA/ECB/OAEPWithSHA-1AndMGF1Padding (1024, 2048)
  RSA/ECB/OAEPWithSHA-256AndMGF1Padding (1024, 2048)

  These transformations are described in the

  Cipher section of the
  Java Cryptography Architecture Standard Algorithm Name Documentation.
  Consult the release documentation for your implementation to see if any
  other transformations are supported."
  (:refer-clojure :only [require comment defn ->])
  (:import [javax.crypto Cipher]))

(def *-encrypt-mode
  "Static Constant.

  Constant used to initialize cipher to encryption mode.

  type: int"
  Cipher/ENCRYPT_MODE)

(def *-decrypt-mode
  "Static Constant.

  Constant used to initialize cipher to decryption mode.

  type: int"
  Cipher/DECRYPT_MODE)

(def *-wrap-mode
  "Static Constant.

  Constant used to initialize cipher to key-wrapping mode.

  type: int"
  Cipher/WRAP_MODE)

(def *-unwrap-mode
  "Static Constant.

  Constant used to initialize cipher to key-unwrapping mode.

  type: int"
  Cipher/UNWRAP_MODE)

(def *-public-key
  "Static Constant.

  Constant used to indicate the to-be-unwrapped key is a `public key`.

  type: int"
  Cipher/PUBLIC_KEY)

(def *-private-key
  "Static Constant.

  Constant used to indicate the to-be-unwrapped key is a `private key`.

  type: int"
  Cipher/PRIVATE_KEY)

(def *-secret-key
  "Static Constant.

  Constant used to indicate the to-be-unwrapped key is a `secret key`.

  type: int"
  Cipher/SECRET_KEY)

(defn *get-instance
  "Returns a Cipher object that implements the specified
   transformation.

    A new Cipher object encapsulating the
   CipherSpi implementation from the specified provider
   is returned.  The specified provider must be registered
   in the security provider list.

    Note that the list of registered providers may be retrieved via
   the Security.getProviders() method.

  transformation - the name of the transformation, e.g., AES/CBC/PKCS5Padding. See the Cipher section in the Java Cryptography Architecture Standard Algorithm Name Documentation for information about standard transformation names. - `java.lang.String`
  provider - the name of the provider. - `java.lang.String`

  returns: a cipher that implements the requested transformation. - `javax.crypto.Cipher`

  throws: java.security.NoSuchAlgorithmException - if transformation is null, empty, in an invalid format, or if a CipherSpi implementation for the specified algorithm is not available from the specified provider."
  ([transformation provider]
    (Cipher/getInstance transformation provider))
  ([transformation]
    (Cipher/getInstance transformation)))

(defn *get-max-allowed-key-length
  "Returns the maximum key length for the specified transformation
   according to the installed JCE jurisdiction policy files. If
   JCE unlimited strength jurisdiction policy files are installed,
   Integer.MAX_VALUE will be returned.
   For more information on default key size in JCE jurisdiction
   policy files, please see Appendix E in the

   Java Cryptography Architecture Reference Guide.

  transformation - the cipher transformation. - `java.lang.String`

  returns: the maximum key length in bits or Integer.MAX_VALUE. - `int`

  throws: java.lang.NullPointerException - if transformation is null."
  ([transformation]
    (Cipher/getMaxAllowedKeyLength transformation)))

(defn *get-max-allowed-parameter-spec
  "Returns an AlgorithmParameterSpec object which contains
   the maximum cipher parameter value according to the
   jurisdiction policy file. If JCE unlimited strength jurisdiction
   policy files are installed or there is no maximum limit on the
   parameters for the specified transformation in the policy file,
   null will be returned.

  transformation - the cipher transformation. - `java.lang.String`

  returns: an AlgorithmParameterSpec which holds the maximum
   value or null. - `java.security.spec.AlgorithmParameterSpec`

  throws: java.lang.NullPointerException - if transformation is null."
  ([transformation]
    (Cipher/getMaxAllowedParameterSpec transformation)))

(defn get-iv
  "Returns the initialization vector (IV) in a new buffer.

   This is useful in the case where a random IV was created,
   or in the context of password-based encryption or
   decryption, where the IV is derived from a user-supplied password.

  returns: the initialization vector in a new buffer, or null if the
   underlying algorithm does not use an IV, or if the IV has not yet
   been set. - `byte[]`"
  ([this]
    (-> this (.getIV))))

(defn get-parameters
  "Returns the parameters used with this cipher.

   The returned parameters may be the same that were used to initialize
   this cipher, or may contain a combination of default and random
   parameter values used by the underlying cipher implementation if this
   cipher requires algorithm parameters but was not initialized with any.

  returns: the parameters used with this cipher, or null if this cipher
   does not use any parameters. - `java.security.AlgorithmParameters`"
  ([this]
    (-> this (.getParameters))))

(defn get-provider
  "Returns the provider of this Cipher object.

  returns: the provider of this Cipher object - `java.security.Provider`"
  ([this]
    (-> this (.getProvider))))

(defn update
  "Continues a multiple-part encryption or decryption operation
   (depending on how this cipher was initialized), processing another data
   part.

   The first inputLen bytes in the input
   buffer, starting at inputOffset inclusive, are processed,
   and the result is stored in the output buffer, starting at
   outputOffset inclusive.

   If the output buffer is too small to hold the result,
   a ShortBufferException is thrown. In this case, repeat this
   call with a larger output buffer. Use
   getOutputSize to determine how big
   the output buffer should be.

   If inputLen is zero, this method returns
   a length of zero.

   Note: this method should be copy-safe, which means the
   input and output buffers can reference
   the same byte array and no unprocessed input data is overwritten
   when the result is copied into the output buffer.

  input - the input buffer - `byte[]`
  input-offset - the offset in input where the input starts - `int`
  input-len - the input length - `int`
  output - the buffer for the result - `byte[]`
  output-offset - the offset in output where the result is stored - `int`

  returns: the number of bytes stored in output - `int`

  throws: java.lang.IllegalStateException - if this cipher is in a wrong state (e.g., has not been initialized)"
  ([this input input-offset input-len output output-offset]
    (-> this (.update input input-offset input-len output output-offset)))
  ([this input input-offset input-len output]
    (-> this (.update input input-offset input-len output)))
  ([this input input-offset input-len]
    (-> this (.update input input-offset input-len)))
  ([this input output]
    (-> this (.update input output)))
  ([this input]
    (-> this (.update input))))

(defn unwrap
  "Unwrap a previously wrapped key.

  wrapped-key - the key to be unwrapped. - `byte[]`
  wrapped-key-algorithm - the algorithm associated with the wrapped key. - `java.lang.String`
  wrapped-key-type - the type of the wrapped key. This must be one of SECRET_KEY, PRIVATE_KEY, or PUBLIC_KEY. - `int`

  returns: the unwrapped key. - `java.security.Key`

  throws: java.lang.IllegalStateException - if this cipher is in a wrong state (e.g., has not been initialized)."
  ([this wrapped-key wrapped-key-algorithm wrapped-key-type]
    (-> this (.unwrap wrapped-key wrapped-key-algorithm wrapped-key-type))))

(defn wrap
  "Wrap a key.

  key - the key to be wrapped. - `java.security.Key`

  returns: the wrapped key. - `byte[]`

  throws: java.lang.IllegalStateException - if this cipher is in a wrong state (e.g., has not been initialized)."
  ([this key]
    (-> this (.wrap key))))

(defn get-exemption-mechanism
  "Returns the exemption mechanism object used with this cipher.

  returns: the exemption mechanism object used with this cipher, or
   null if this cipher does not use any exemption mechanism. - `javax.crypto.ExemptionMechanism`"
  ([this]
    (-> this (.getExemptionMechanism))))

(defn get-algorithm
  "Returns the algorithm name of this Cipher object.

   This is the same name that was specified in one of the
   getInstance calls that created this Cipher
   object..

  returns: the algorithm name of this Cipher object. - `java.lang.String`"
  ([this]
    (-> this (.getAlgorithm))))

(defn init
  "Initializes this cipher with a key, a set of algorithm
   parameters, and a source of randomness.

   The cipher is initialized for one of the following four operations:
   encryption, decryption, key wrapping or  key unwrapping, depending
   on the value of opmode.

   If this cipher requires any algorithm parameters and
   params is null, the underlying cipher implementation is
   supposed to generate the required parameters itself (using
   provider-specific default or random values) if it is being
   initialized for encryption or key wrapping, and raise an
   InvalidAlgorithmParameterException if it is being
   initialized for decryption or key unwrapping.
   The generated parameters can be retrieved using
   getParameters or
   getIV (if the parameter is an IV).

   If this cipher requires algorithm parameters that cannot be
   derived from the input parameters, and there are no reasonable
   provider-specific default values, initialization will
   necessarily fail.

   If this cipher (including its underlying feedback or padding scheme)
   requires any random bytes (e.g., for parameter generation), it will get
   them from random.

   Note that when a Cipher object is initialized, it loses all
   previously-acquired state. In other words, initializing a Cipher is
   equivalent to creating a new instance of that Cipher and initializing
   it.

  opmode - the operation mode of this cipher (this is one of the following: ENCRYPT_MODE, DECRYPT_MODE, WRAP_MODE or UNWRAP_MODE) - `int`
  key - the encryption key - `java.security.Key`
  params - the algorithm parameters - `java.security.spec.AlgorithmParameterSpec`
  random - the source of randomness - `java.security.SecureRandom`

  throws: java.security.InvalidKeyException - if the given key is inappropriate for initializing this cipher, or its keysize exceeds the maximum allowable keysize (as determined from the configured jurisdiction policy files)."
  ([this opmode key params random]
    (-> this (.init opmode key params random)))
  ([this opmode key random]
    (-> this (.init opmode key random)))
  ([this opmode key]
    (-> this (.init opmode key))))

(defn update-aad
  "Continues a multi-part update of the Additional Authentication
   Data (AAD), using a subset of the provided buffer.

   Calls to this method provide AAD to the cipher when operating in
   modes such as AEAD (GCM/CCM).  If this cipher is operating in
   either GCM or CCM mode, all AAD must be supplied before beginning
   operations on the ciphertext (via the update
   and doFinal methods).

  src - the buffer containing the AAD - `byte[]`
  offset - the offset in src where the AAD input starts - `int`
  len - the number of AAD bytes - `int`

  throws: java.lang.IllegalArgumentException - if the src byte array is null, or the offset or length is less than 0, or the sum of the offset and len is greater than the length of the src byte array"
  ([this src offset len]
    (-> this (.updateAAD src offset len)))
  ([this src]
    (-> this (.updateAAD src))))

(defn get-block-size
  "Returns the block size (in bytes).

  returns: the block size (in bytes), or 0 if the underlying algorithm is
   not a block cipher - `int`"
  ([this]
    (-> this (.getBlockSize))))

(defn get-output-size
  "Returns the length in bytes that an output buffer would need to be in
   order to hold the result of the next update or
   doFinal operation, given the input length
   inputLen (in bytes).

   This call takes into account any unprocessed (buffered) data from a
   previous update call, padding, and AEAD tagging.

   The actual output length of the next update or
   doFinal call may be smaller than the length returned by
   this method.

  input-len - the input length (in bytes) - `int`

  returns: the required output buffer size (in bytes) - `int`

  throws: java.lang.IllegalStateException - if this cipher is in a wrong state (e.g., has not yet been initialized)"
  ([this input-len]
    (-> this (.getOutputSize input-len))))

(defn do-final
  "Encrypts or decrypts data in a single-part operation, or finishes a
   multiple-part operation. The data is encrypted or decrypted,
   depending on how this cipher was initialized.

   The first inputLen bytes in the input
   buffer, starting at inputOffset inclusive, and any input
   bytes that may have been buffered during a previous
   update operation, are processed, with padding
   (if requested) being applied.
   If an AEAD mode such as GCM/CCM is being used, the authentication
   tag is appended in the case of encryption, or verified in the
   case of decryption.
   The result is stored in the output buffer, starting at
   outputOffset inclusive.

   If the output buffer is too small to hold the result,
   a ShortBufferException is thrown. In this case, repeat this
   call with a larger output buffer. Use
   getOutputSize to determine how big
   the output buffer should be.

   Upon finishing, this method resets this cipher object to the state
   it was in when previously initialized via a call to init.
   That is, the object is reset and available to encrypt or decrypt
   (depending on the operation mode that was specified in the call to
   init) more data.

   Note: if any exception is thrown, this cipher object may need to
   be reset before it can be used again.

   Note: this method should be copy-safe, which means the
   input and output buffers can reference
   the same byte array and no unprocessed input data is overwritten
   when the result is copied into the output buffer.

  input - the input buffer - `byte[]`
  input-offset - the offset in input where the input starts - `int`
  input-len - the input length - `int`
  output - the buffer for the result - `byte[]`
  output-offset - the offset in output where the result is stored - `int`

  returns: the number of bytes stored in output - `int`

  throws: java.lang.IllegalStateException - if this cipher is in a wrong state (e.g., has not been initialized)"
  ([this input input-offset input-len output output-offset]
    (-> this (.doFinal input input-offset input-len output output-offset)))
  ([this input input-offset input-len output]
    (-> this (.doFinal input input-offset input-len output)))
  ([this input input-offset input-len]
    (-> this (.doFinal input input-offset input-len)))
  ([this output output-offset]
    (-> this (.doFinal output output-offset)))
  ([this input]
    (-> this (.doFinal input)))
  ([this]
    (-> this (.doFinal))))

