(ns javax.naming.Context
  "This interface represents a naming context, which
  consists of a set of name-to-object bindings.
  It contains methods for examining and updating these bindings.

  Names
  Each name passed as an argument to a Context method is relative
  to that context.  The empty name is used to name the context itself.
  A name parameter may never be null.

  Most of the methods have overloaded versions with one taking a
  Name parameter and one taking a String.
  These overloaded versions are equivalent in that if
  the Name and String parameters are just
  different representations of the same name, then the overloaded
  versions of the same methods behave the same.
  In the method descriptions below, only one version is fully documented.
  The second version instead has a link to the first:  the same
  documentation applies to both.

  For systems that support federation, String name arguments to
  Context methods are composite names. Name arguments that are
  instances of CompositeName are treated as composite names,
  while Name arguments that are not instances of
  CompositeName are treated as compound names (which might be
  instances of CompoundName or other implementations of compound
  names). This allows the results of NameParser.parse() to be used as
  arguments to the Context methods.
  Prior to JNDI 1.2, all name arguments were treated as composite names.

  Furthermore, for systems that support federation, all names returned
  in a NamingEnumeration
  from list() and listBindings() are composite names
  represented as strings.
  See CompositeName for the string syntax of names.

  For systems that do not support federation, the name arguments (in
  either Name or String forms) and the names returned in
  NamingEnumeration may be names in their own namespace rather than
  names in a composite namespace, at the discretion of the service
  provider.

 Exceptions
  All the methods in this interface can throw a NamingException or
  any of its subclasses. See NamingException and their subclasses
  for details on each exception.

 Concurrent Access
  A Context instance is not guaranteed to be synchronized against
  concurrent access by multiple threads.  Threads that need to access
  a single Context instance concurrently should synchronize amongst
  themselves and provide the necessary locking.  Multiple threads
  each manipulating a different Context instance need not
  synchronize.  Note that the lookup
  method, when passed an empty name, will return a new Context instance
  representing the same naming context.

  For purposes of concurrency control,
  a Context operation that returns a NamingEnumeration is
  not considered to have completed while the enumeration is still in
  use, or while any referrals generated by that operation are still
  being followed.


 Parameters
  A Name parameter passed to any method of the
  Context interface or one of its subinterfaces
  will not be modified by the service provider.
  The service provider may keep a reference to it
  for the duration of the operation, including any enumeration of the
  method's results and the processing of any referrals generated.
  The caller should not modify the object during this time.
  A Name returned by any such method is owned by the caller.
  The caller may subsequently modify it; the service provider may not.


 Environment Properties

  JNDI applications need a way to communicate various preferences
  and properties that define the environment in which naming and
  directory services are accessed. For example, a context might
  require specification of security credentials in order to access
  the service. Another context might require that server configuration
  information be supplied. These are referred to as the environment
  of a context. The Context interface provides methods for
  retrieving and updating this environment.

  The environment is inherited from the parent context as
  context methods proceed from one context to the next. Changes to
  the environment of one context do not directly affect those
  of other contexts.

  It is implementation-dependent when environment properties are used
  and/or verified for validity.  For example, some of the
  security-related properties are used by service providers to `log in`
  to the directory.  This login process might occur at the time the
  context is created, or the first time a method is invoked on the
  context.  When, and whether this occurs at all, is
  implementation-dependent.  When environment properties are added or
  removed from the context, verifying the validity of the changes is again
  implementation-dependent. For example, verification of some properties
  might occur at the time the change is made, or at the time the next
  operation is performed on the context, or not at all.

  Any object with a reference to a context may examine that context's
  environment.  Sensitive information such as clear-text
  passwords should not be stored there unless the implementation is
  known to protect it.



 Resource Files

  To simplify the task of setting up the environment
  required by a JNDI application,
  application components and service providers may be distributed
  along with resource files.
  A JNDI resource file is a file in the properties file format (see
  java.util.Properties),
  containing a list of key/value pairs.
  The key is the name of the property (e.g. `java.naming.factory.object`)
  and the value is a string in the format defined
  for that property.  Here is an example of a JNDI resource file:

  java.naming.factory.object=com.sun.jndi.ldap.AttrsToCorba:com.wiz.from.Person
  java.naming.factory.state=com.sun.jndi.ldap.CorbaToAttrs:com.wiz.from.Person
  java.naming.factory.control=com.sun.jndi.ldap.ResponseControlFactory


  The JNDI class library reads the resource files and makes the property
  values freely available.  Thus JNDI resource files should be considered
  to be `world readable`, and sensitive information such as clear-text
  passwords should not be stored there.

  There are two kinds of JNDI resource files:
  provider and application.

  Provider Resource Files

  Each service provider has an optional resource that lists properties
  specific to that provider.  The name of this resource is:

  [prefix/]jndiprovider.properties

  where prefix is
  the package name of the provider's context implementation(s),
  with each period (`.`) converted to a slash (`/`).

  For example, suppose a service provider defines a context
  implementation with class name com.sun.jndi.ldap.LdapCtx.
  The provider resource for this provider is named
  com/sun/jndi/ldap/jndiprovider.properties.  If the class is
  not in a package, the resource's name is simply
  jndiprovider.properties.



  Certain methods in the JNDI class library make use of the standard
  JNDI properties that specify lists of JNDI factories:

  java.naming.factory.object
  java.naming.factory.state
  java.naming.factory.control
  java.naming.factory.url.pkgs

  The JNDI library will consult the provider resource file
  when determining the values of these properties.
  Properties other than these may be set in the provider
  resource file at the discretion of the service provider.
  The service provider's documentation should clearly state which
  properties are allowed; other properties in the file will be ignored.

  Application Resource Files

  When an application is deployed, it will generally have several
  codebase directories and JARs in its classpath.  Similarly, when an
  applet is deployed, it will have a codebase and archives specifying
  where to find the applet's classes.  JNDI locates (using
  ClassLoader.getResources())
  all application resource files named jndi.properties
  in the classpath.
  In addition, if the file java.home/lib/jndi.properties
  exists and is readable,
  JNDI treats it as an additional application resource file.
  (java.home indicates the
  directory named by the java.home system property.)
  All of the properties contained in these files are placed
  into the environment of the initial context.  This environment
  is then inherited by other contexts.


  For each property found in more than one application resource file,
  JNDI uses the first value found or, in a few cases where it makes
  sense to do so, it concatenates all of the values (details are given
  below).
  For example, if the `java.naming.factory.object` property is found in
  three jndi.properties resource files, the
  list of object factories is a concatenation of the property
  values from all three files.
  Using this scheme, each deployable component is responsible for
  listing the factories that it exports.  JNDI automatically
  collects and uses all of these export lists when searching for factory
  classes.

  Search Algorithm for Properties

  When JNDI constructs an initial context, the context's environment
  is initialized with properties defined in the environment parameter
  passed to the constructor, the system properties, the applet parameters,
  and the application resource files.  See
  InitialContext
  for details.
  This initial environment is then inherited by other context instances.


  When the JNDI class library needs to determine
  the value of a property, it does so by merging
  the values from the following two sources, in order:

  The environment of the context being operated on.
  The provider resource file (jndiprovider.properties)
  for the context being operated on.

  For each property found in both of these two sources,
  JNDI determines the property's value as follows.  If the property is
  one of the standard JNDI properties that specify a list of JNDI
  factories (listed above), the values are
  concatenated into a single colon-separated list.  For other
  properties, only the first value found is used.


  When a service provider needs to determine the value of a property,
  it will generally take that value directly from the environment.
  A service provider may define provider-specific properties
  to be placed in its own provider resource file.  In that
  case it should merge values as described in the previous paragraph.


  In this way, each service provider developer can specify a list of
  factories to use with that service provider. These can be modified by
  the application resources specified by the deployer of the application
  or applet, which in turn can be modified by the user."
  (:refer-clojure :only [require comment defn ->])
  (:import [javax.naming Context]))

(defn destroy-subcontext
  "Destroys the named context and removes it from the namespace.
   Any attributes associated with the name are also removed.
   Intermediate contexts are not destroyed.

    This method is idempotent.
   It succeeds even if the terminal atomic name
   is not bound in the target context, but throws
   NameNotFoundException
   if any of the intermediate contexts do not exist.

    In a federated naming system, a context from one naming system
   may be bound to a name in another.  One can subsequently
   look up and perform operations on the foreign context using a
   composite name.  However, an attempt destroy the context using
   this composite name will fail with
   NotContextException, because the foreign context is not
   a `subcontext` of the context in which it is bound.
   Instead, use unbind() to remove the
   binding of the foreign context.  Destroying the foreign context
   requires that the destroySubcontext() be performed
   on a context from the foreign context's `native` naming system.

  name - the name of the context to be destroyed; may not be empty - `javax.naming.Name`

  throws: javax.naming.NameNotFoundException - if an intermediate context does not exist"
  ([this name]
    (-> this (.destroySubcontext name))))

(defn unbind
  "Unbinds the named object.
   Removes the terminal atomic name in name
   from the target context--that named by all but the terminal
   atomic part of name.

    This method is idempotent.
   It succeeds even if the terminal atomic name
   is not bound in the target context, but throws
   NameNotFoundException
   if any of the intermediate contexts do not exist.

    Any attributes associated with the name are removed.
   Intermediate contexts are not changed.

  name - the name to unbind; may not be empty - `javax.naming.Name`

  throws: javax.naming.NameNotFoundException - if an intermediate context does not exist"
  ([this name]
    (-> this (.unbind name))))

(defn compose-name
  "Composes the name of this context with a name relative to
   this context.
   Given a name (name) relative to this context, and
   the name (prefix) of this context relative to one
   of its ancestors, this method returns the composition of the
   two names using the syntax appropriate for the naming
   system(s) involved.  That is, if name names an
   object relative to this context, the result is the name of the
   same object, but relative to the ancestor context.  None of the
   names may be null.

   For example, if this context is named `wiz.com` relative
   to the initial context, then


    composeName(`east`, `wiz.com`)
   might return `east.wiz.com`.
   If instead this context is named `org/research`, then


    composeName(`user/jane`, `org/research`)
   might return `org/research/user/jane` while


    composeName(`user/jane`, `research`)
   returns `research/user/jane`.

  name - a name relative to this context - `javax.naming.Name`
  prefix - the name of this context relative to one of its ancestors - `javax.naming.Name`

  returns: the composition of prefix and name - `javax.naming.Name`

  throws: javax.naming.NamingException - if a naming exception is encountered"
  ([this name prefix]
    (-> this (.composeName name prefix))))

(defn rename
  "Binds a new name to the object bound to an old name, and unbinds
   the old name.  Both names are relative to this context.
   Any attributes associated with the old name become associated
   with the new name.
   Intermediate contexts of the old name are not changed.

  old-name - the name of the existing binding; may not be empty - `javax.naming.Name`
  new-name - the name of the new binding; may not be empty - `javax.naming.Name`

  throws: javax.naming.NameAlreadyBoundException - if newName is already bound"
  ([this old-name new-name]
    (-> this (.rename old-name new-name))))

(defn list
  "Enumerates the names bound in the named context, along with the
   class names of objects bound to them.
   The contents of any subcontexts are not included.

    If a binding is added to or removed from this context,
   its effect on an enumeration previously returned is undefined.

  name - the name of the context to list - `javax.naming.Name`

  returns: an enumeration of the names and class names of the
            bindings in this context.  Each element of the
            enumeration is of type NameClassPair. - `javax.naming.NamingEnumeration<javax.naming.NameClassPair>`

  throws: javax.naming.NamingException - if a naming exception is encountered"
  ([this name]
    (-> this (.list name))))

(defn add-to-environment
  "Adds a new environment property to the environment of this
   context.  If the property already exists, its value is overwritten.
   See class description for more details on environment properties.

  prop-name - the name of the environment property to add; may not be null - `java.lang.String`
  prop-val - the value of the property to add; may not be null - `java.lang.Object`

  returns: the previous value of the property, or null if the property was
            not in the environment before - `java.lang.Object`

  throws: javax.naming.NamingException - if a naming exception is encountered"
  ([this prop-name prop-val]
    (-> this (.addToEnvironment prop-name prop-val))))

(defn bind
  "Binds a name to an object.
   All intermediate contexts and the target context (that named by all
   but terminal atomic component of the name) must already exist.

  name - the name to bind; may not be empty - `javax.naming.Name`
  obj - the object to bind; possibly null - `java.lang.Object`

  throws: javax.naming.NameAlreadyBoundException - if name is already bound"
  ([this name obj]
    (-> this (.bind name obj))))

(defn list-bindings
  "Enumerates the names bound in the named context, along with the
   objects bound to them.
   The contents of any subcontexts are not included.

    If a binding is added to or removed from this context,
   its effect on an enumeration previously returned is undefined.

  name - the name of the context to list - `javax.naming.Name`

  returns: an enumeration of the bindings in this context.
            Each element of the enumeration is of type
            Binding. - `javax.naming.NamingEnumeration<javax.naming.Binding>`

  throws: javax.naming.NamingException - if a naming exception is encountered"
  ([this name]
    (-> this (.listBindings name))))

(defn lookup
  "Retrieves the named object.
   If name is empty, returns a new instance of this context
   (which represents the same naming context as this context, but its
   environment may be modified independently and it may be accessed
   concurrently).

  name - the name of the object to look up - `javax.naming.Name`

  returns: the object bound to name - `java.lang.Object`

  throws: javax.naming.NamingException - if a naming exception is encountered"
  ([this name]
    (-> this (.lookup name))))

(defn remove-from-environment
  "Removes an environment property from the environment of this
   context.  See class description for more details on environment
   properties.

  prop-name - the name of the environment property to remove; may not be null - `java.lang.String`

  returns: the previous value of the property, or null if the property was
            not in the environment - `java.lang.Object`

  throws: javax.naming.NamingException - if a naming exception is encountered"
  ([this prop-name]
    (-> this (.removeFromEnvironment prop-name))))

(defn create-subcontext
  "Creates and binds a new context.
   Creates a new context with the given name and binds it in
   the target context (that named by all but terminal atomic
   component of the name).  All intermediate contexts and the
   target context must already exist.

  name - the name of the context to create; may not be empty - `javax.naming.Name`

  returns: the newly created context - `javax.naming.Context`

  throws: javax.naming.NameAlreadyBoundException - if name is already bound"
  ([this name]
    (-> this (.createSubcontext name))))

(defn get-name-parser
  "Retrieves the parser associated with the named context.
   In a federation of namespaces, different naming systems will
   parse names differently.  This method allows an application
   to get a parser for parsing names into their atomic components
   using the naming convention of a particular naming system.
   Within any single naming system, NameParser objects
   returned by this method must be equal (using the equals()
   test).

  name - the name of the context from which to get the parser - `javax.naming.Name`

  returns: a name parser that can parse compound names into their atomic
            components - `javax.naming.NameParser`

  throws: javax.naming.NamingException - if a naming exception is encountered"
  ([this name]
    (-> this (.getNameParser name))))

(defn get-environment
  "Retrieves the environment in effect for this context.
   See class description for more details on environment properties.

    The caller should not make any changes to the object returned:
   their effect on the context is undefined.
   The environment of this context may be changed using
   addToEnvironment() and removeFromEnvironment().

  returns: the environment of this context; never null - `java.util.Hashtable<?,?>`

  throws: javax.naming.NamingException - if a naming exception is encountered"
  ([this]
    (-> this (.getEnvironment))))

(defn rebind
  "Binds a name to an object, overwriting any existing binding.
   All intermediate contexts and the target context (that named by all
   but terminal atomic component of the name) must already exist.

    If the object is a DirContext, any existing attributes
   associated with the name are replaced with those of the object.
   Otherwise, any existing attributes associated with the name remain
   unchanged.

  name - the name to bind; may not be empty - `javax.naming.Name`
  obj - the object to bind; possibly null - `java.lang.Object`

  throws: javax.naming.directory.InvalidAttributesException - if object did not supply all mandatory attributes"
  ([this name obj]
    (-> this (.rebind name obj))))

(defn close
  "Closes this context.
   This method releases this context's resources immediately, instead of
   waiting for them to be released automatically by the garbage collector.

    This method is idempotent:  invoking it on a context that has
   already been closed has no effect.  Invoking any other method
   on a closed context is not allowed, and results in undefined behaviour.

  throws: javax.naming.NamingException - if a naming exception is encountered"
  ([this]
    (-> this (.close))))

(defn get-name-in-namespace
  "Retrieves the full name of this context within its own namespace.

    Many naming services have a notion of a `full name` for objects
   in their respective namespaces.  For example, an LDAP entry has
   a distinguished name, and a DNS record has a fully qualified name.
   This method allows the client application to retrieve this name.
   The string returned by this method is not a JNDI composite name
   and should not be passed directly to context methods.
   In naming systems for which the notion of full name does not
   make sense, OperationNotSupportedException is thrown.

  returns: this context's name in its own namespace; never null - `java.lang.String`

  throws: javax.naming.OperationNotSupportedException - if the naming system does not have the notion of a full name"
  ([this]
    (-> this (.getNameInNamespace))))

(defn lookup-link
  "Retrieves the named object, following links except
   for the terminal atomic component of the name.
   If the object bound to name is not a link,
   returns the object itself.

  name - the name of the object to look up - `javax.naming.Name`

  returns: the object bound to name, not following the
            terminal link (if any). - `java.lang.Object`

  throws: javax.naming.NamingException - if a naming exception is encountered"
  ([this name]
    (-> this (.lookupLink name))))

