(ns clojusc.twig.component
  "Logging component to be used with the Clojure Component library."
  (:require [clojusc.twig :as logger]
            [com.stuartsierra.component :as component]
            [taoensso.timbre :as log]
            [trifl.java]))

(defn get-log-formatter
  [color?]
  (if (= color? true)
    logger/color-log-formatter
    logger/no-color-log-formatter))

(defn set-level
  [color? logger-cfg]
  (let [default-formatter (get-log-formatter color?)
        formatter (:formatter logger-cfg)
        log-level (:level logger-cfg)
        namespaces (:namespaces logger-cfg)]
    (log/debugf
      "Using log-level %s for namespace(s) %s ..."
      log-level namespaces)
    (logger/set-level!
      namespaces
      log-level
      (if (nil? formatter)
        default-formatter
        formatter))))

(defrecord LoggerComponent [loggers-cfg color?]
  component/Lifecycle

  (start [component]
    (log/info "Setting up logging component ...")

    ; (logger/set-level! namespaces log-level formatter)
    (log/debug "Using color?" color?)
    (log/trace "Loggers config:" loggers-cfg)
    (doall
      (map (partial set-level color?) loggers-cfg))
    (log/debug "Successfully configured logging.")
    component)

  (stop [component]
    (log/info "Tearing down logging component ...")
    component))

(defn new-logger [loggers-cfg & {:keys [color?]}]
  "Constructor for logger component.

  This function takes two arguments:
   * a data structure for configuring logging for one or more namespaces, and
   * a keyword / boolean pair for declaring whether color output should be
     used.

  The data structure is of the following form:
  ```
  [{<namespaces> <log level> <optional formatter>}]
  ```
  For example, the following would be a valid loggers configuration data
  structure:
  ```
  [{:namespaces 'my-ns :level :info}
   {:namespaces '[your-ns other-ns] :level :debug}
   {:namespaces '[more-ns] :level trace :formatter #'my-fmtr}]
  ```
  That particular configuration will set three differentn log levels, one of
  which will be for two separate namespaces together."
  (->LoggerComponent loggers-cfg color?))

(defn init
  "This init function is only intended to be used when setting up the
  logging component as the only element in the system (e.g., for testing/
  troubleshooting). When the logging component is used in a larger system,
  that system's `init` function will do something similar, but utilize
  whatever configuration approach that system uses in order to pass the
  proper arguments to the `new-logger` constructor."
  []
  (component/system-map
    :log (component/using
           (new-logger [{:namespaces '[clojusc com.stuartsierra.component]
                         :level :debug}]
                       :color? true)
           [])))

(defn stop
  [system component-key]
  (->> system
       (component-key)
       (component/stop)
       (assoc system component-key)))

(defn start
  "This variadic function is overloaded to support two cases:
  1) with no arguments, where it is used to start the system, and
  2) with the standard arguments used to satisfy the Component library
     interface.

  Note that this is non-standard usage, and when the logging component is
  used in a complete system, the 2-artiy function will be used in one
  namespace (the namespace that is responsibile for implementing the Component
  library system interface) and the functionality entailed in the 0-arity
  function below will be implemented at the application-level or wherever
  the system as a whole is managed/started."
  ([]
    (let [system (init)]
      (component/start system)
      (trifl.java/add-shutdown-handler #(component/stop system))))
  ([system component-key]
    (->> system
         (component-key)
         (component/start)
         (assoc system component-key))))

(defn restart
  [system component-key]
  (-> system
      (stop component-key)
      (start component-key)))
