(ns migrations.dml
  (:require [clojure.java.jdbc :as j]
            [cheshire.core :as json]
            [clj-yaml.core :as yaml]
            [clojure.tools.logging :refer [log]])
  (:import org.postgresql.util.PGobject))

(def config*
  (-> "db.yaml"
    clojure.java.io/resource
    slurp
    yaml/parse-string))

(defn config [host]
  (assoc config* :host host))

(extend-protocol j/IResultSetReadColumn
  PGobject
  (result-set-read-column [pgobj metadata idx]
    (let [type  (.getType pgobj)
          value (.getValue pgobj)]
      (case type
        "jsonb" (json/parse-string value true)
        :else value))))

(defn value-to-json-pgobject [value]
  (doto (PGobject.)
    (.setType "jsonb")
      (.setValue (json/generate-string value))))

(extend-protocol j/ISQLValue
  clojure.lang.IPersistentMap
  (sql-value [value] (value-to-json-pgobject value))

  clojure.lang.IPersistentVector
  (sql-value [value] (value-to-json-pgobject value)))

(defn rollback [config]
  (j/with-db-transaction [conn config]
    (j/delete! conn :program [])
    (j/delete! conn :test_run [])
    (j/delete! conn :log_entry [])
    (j/delete! conn :checkpoint [])
    (j/delete! conn :module [])
    (j/delete! conn :raw_result [])
    (j/delete! conn :config_profile [])))

(defn programs-from-linen-dir
  [programs f & [parent-name program]]
  ;; for each program-type
  ;;   for each program
  ;;     for each config in the config dir
  ;;       insert program as:
  ;;
  ;; name: prg dir parent name, ".", prg dir name
  ;; data: map of prg data
  ;; configs: list of configs
  ;; name: filename minus ".yaml"
  ;; data: json serialization of file contents
  ;; program_name: program directory parent name, ".",
  ;;     program directory name
  (if (.exists f)
    (if (.isDirectory f)
      ;; contains config dir? get program name
      (let [prg (if-not (nil? program)
                  program
                  (when
                    (some
                      #(and (.isDirectory %)
                            (= (.getName %) "config"))
                      (.listFiles f))
                    {:name (str parent-name "." (.getName f))
                     :data (->> (.listFiles f)
                             (filter #(not (.isDirectory %)))
                             first
                             slurp
                             yaml/parse-string
                             )}))]
        (map #(programs-from-linen-dir
                programs % (.getName f) prg)
             (.listFiles f)))
      (if (or (nil? parent-name)
              (not= parent-name "config"))
        programs
        (reduce
          (fn [configs conf]
            (assoc program
             :configs
             (conj (:configs program)
               {:name (-> f
                       .getName
                       (clojure.string/replace #".yaml$" ""))
                :program_name (:name program)
                :data conf})))
          programs
          (-> f slurp yaml/parse-all))))
    programs))

(defn migrate [db-config]
  (let [modules (-> "linen/modules"
                    clojure.java.io/resource
                    clojure.java.io/file
                    .listFiles)]
    (doseq [m modules]
      (let [nm (-> m .getName (clojure.string/replace #".yaml$" ""))]
        (try
          (j/with-db-transaction [conn db-config]
            (j/insert! conn :module
              {:name nm
               :data (-> m slurp yaml/parse-string)})
            (log :info (str "module \"" nm "\" inserted.")))
          (catch org.postgresql.util.PSQLException psqle
            (log :warn (str "module \""
                            nm
                            "\" not inserted: " (.getMessage psqle))))))))
  (let [programs
        (flatten
          (reduce
            programs-from-linen-dir
            []
            (.listFiles
              (clojure.java.io/file
                (clojure.java.io/resource "linen")))))]
    (doseq [prg programs]
      (try
        (j/with-db-transaction [conn db-config]
          (j/insert! conn :program
            (dissoc prg :configs))
          (doseq [conf (:configs prg)]
            (j/insert! conn :config_profile
              conf))
          (log :info (str "program \"" (:name prg) "\" inserted.")))
        (catch org.postgresql.util.PSQLException psqle
          (log :warn (str "program \""
                          (:name prg)
                          "\" not inserted: " (.getMessage psqle)))))
      )))

(defn -main [host & [direction]]
  (if (= direction "down")
    (rollback (config host))
    (migrate (config host))))

