(ns co.nclk.laundry.core
  (:require [ring.adapter.jetty :as jetty]
            [ring.middleware.defaults :refer [wrap-defaults
                                              api-defaults]]
            [compojure.core :refer :all]
            [compojure.route :as route]
            [co.nclk.laundry.actions :as actions]
            [co.nclk.laundry.models.common :as models]
            [co.nclk.laundry.middleware :refer [wrap-json
                                                wrap-json-body
                                                wrap-api-exception]]
            [co.nclk.laundry.routes.common :refer :all]
            [co.nclk.laundry.routes.config-profiles :refer :all]
            )
  (:import (java.net NetworkInterface Inet4Address)))

(defn ip-filter [inet]
  (and (.isUp inet)
       (not (.isVirtual inet))
       (not (.isLoopback inet))))

(defn ip-extract [netinf]
  (let [inets (enumeration-seq (.getInetAddresses netinf))]
   (map #(vector (.getHostAddress %1) %2) (filter #(instance? Inet4Address %) inets ) (repeat (.getName netinf)))))

(defn ips []
  (let [ifc (NetworkInterface/getNetworkInterfaces)]
     (mapcat ip-extract (filter ip-filter (enumeration-seq ifc)))))

(defn controller
  [api-base]
  {"/config-profiles"
   {:relation :config_profile
    :description
    (str "Config profiles are resources that persist useful "
         "configurations "
         "for a particular test program. When used with the "
         "`/actions/run` endpoint, each config from the given "
         "list is `assoc`ed into the previous one in sequence, "
         "providing a mechanism for overriding defaults.")
    :related-resources {:program [(str api-base "/programs/%s")
                                  :program_name]}
    :filter-keys #{:name :program_name}
    :insert-keys #{:name :program_name :data}
    :update-keys #{:name :program_name :data}
    :delete-keys #{:name :program_name}
    :individual-href [(str api-base "/config-profiles/%s/%s")
                      :program_name :name]
    :routes [{:path "/"
              :methods #{:get :post}
              :collection? true}
             {:path "/:program_name/"
              :methods #{:get}
              :collection? true}
             {:path "/:program_name/:name"
              :methods #{:get :put :delete}}
             ]}
   "/programs"
   {:relation :program
    :description "A test program written in the flax/linen paradigm."
    :individual-href [(str api-base "/programs/%s")
                      :name]
    :related-resources {:config-profiles
                        [(str api-base "/config-profiles/%s/")
                         :name]
                        :test-runs
                        [(str api-base "/test-runs/?program_name=%s")
                         :name]}
    :filter-keys #{:name}
    :insert-keys #{:name :data}
    :update-keys #{:name :data}
    :routes [{:path "/"
              :methods #{:get :post}
              :collection? true}
             {:path "/:name"
              :methods #{:get :put :delete}}]}
   "/modules"
   {:relation :module
    :description (str "A module written in the flax/linen paradigm "
                      "to be used with linen programs.")
    :individual-href [(str api-base "/modules/%s")
                      :name]
    :filter-keys #{:name}
    :routes [{:path "/"
              :methods #{:get :post}
              :collection? true}
             {:path "/:name"
              :methods #{:get :put :delete}}]}
   "/test-runs"
   {:relation :test_run
    :description (str "Holds data specific to an individual run of a "
                      "program together with its configuration/"
                      "overrides.")
    :individual-href [(str api-base "/test-runs/%s")
                      :id]
    :related-resources {:program
                        [(str api-base "/programs/%s")
                         :program_name]
                        :checkpoints
                        [(str api-base "/checkpoints/?test_run_id=%s")
                         :id]
                        :log-entries
                        [(str api-base "/log-entries/?test_run_id=%s")
                         :id]
                        :raw-result
                        [(str api-base "/raw-results/%s")
                         :id]
                        }
    :filter-keys #{:id|uuid :program_name :status
                   :seed|int :num_failures|int :num_checkpoints|int}
    :routes [{:path "/"
              :methods #{:get}
              :collection? true}
             {:path "/:id"
              :methods #{:get :delete}}]}
   "/raw-results"
   {:relation :raw_result
    :description (str "The fully \"realized\" test program (useful in "
                      "rare cases for debugging.")
    :individual-href [(str api-base "/raw-results/%s")
                      :test_run_id]
    :related-resources {:test-run
                        [(str api-base "/test-runs/%s")
                         :test_run_id]}
    :filter-keys #{:test_run_id|uuid}
    :routes [{:path "/"
              :methods #{:get}
              :collection? true}
             {:path "/:test_run_id"
              :methods #{:get :delete}}]}
   "/checkpoints"
   {:relation :checkpoint
    :description (str "Individual linen checkpoint, which represents "
                      "an \"atomic\" test or subroutine.")
    :individual-href [(str api-base "/checkpoints/%s")
                      :id]
    :related-resources {:test-run
                        [(str api-base "/test-runs/%s")
                         :test_run_id]}
    :filter-keys #{:test_run_id|uuid}
    :routes [{:path "/"
              :methods #{:get}
              :collection? true}
             {:path "/:id"
              :methods #{:get}}]}
   "/log-entries"
   {:relation :log_entry
    :description "Individual log lines from test-runs."
    :individual-href [(str api-base "/log-entries/%s")
                      :id]
    :related-resources {:test-run
                        [(str api-base "/test-runs/%s")
                         :test_run_id]}
    :filter-keys #{:test_run_id|uuid :id|int :level}
    :insert-keys #{}
    :delete-keys #{:test_run_id|uuid :id|int :level}
    :routes [{:path "/"
              :methods #{:get :delete}
              :collection? true}
             {:path "/:id"
              :methods #{:get :delete}}]}
   })

(defn base
  [api-base]
  (GET "/" request
    (let [controller
          (->> api-base
            controller
            (map (fn [[k v]]
                   [k (assoc v :href
                               (resource-uri
                                 (-> request :scheme name)
                                 (-> request :headers (get "host"))
                                 (str api-base k "/")))]))
            (into {}))]
      
      {:status 200 :body controller})))

(defn actions
  [api-base]
  (context "/actions" []

    (GET "/" request
      {:status 200
       :body
       [{:name "run"
         :description (str "Runs a test program against "
                           "a set of config profiles")
         :parameters {:program-name {:type "string"
                                     :required true}
                      :config-profiles {:type "list<config-profile-name>"
                                        :default ["default"]
                                        :required false}}
         :returns {:test_run_id {:type "uuid"}}}]})
    
    (POST "/run" request
      (let [data (:body request)
            config-profiles (:config_profiles data ["default"])
            prg-name (:program_name data)
            prg (models/ett :program :filters {:name prg-name})
            [configs _] (models/samling
                          :config_profile
                          :filters {:program_name prg-name
                                    :name config-profiles})
            configs (conj configs
                      {:name "__$$$__"
                       :data (:env data)})
            resp (actions/run
                   prg
                   configs
                   (resource-uri
                     (-> request :scheme name)
                     (str "localhost"
                          ":"
                          (or (System/getenv "PORT") "80"))
                     #_(-> request :headers
                                 (get "host"))
                     api-base))]
        (if (clojure.string/blank? (:error resp))
          {:status 201
           :body {:test_run_id resp}}
          {:status 400
           :body (:error resp)})))
    ))

(defn api [api-base]
  (context api-base []
    (base api-base)
    (actions api-base)
    (context-compile (controller api-base))
    (route/not-found nil)))

(defn laundry [api-base]
  (-> (routes (api api-base))
    wrap-json-body
    (wrap-defaults api-defaults)
    wrap-api-exception
    wrap-json
    ))

