(ns co.nclk.laundry.actions
  (:require [clojure.java.jdbc :as j]
            [clj-yaml.core :as yaml]
            [cheshire.core :as json]
            [clojure.tools.logging.impl :as logp]
            [clojure.tools.logging :refer [log *logger-factory*]]
            [clojure.core.async :as a]
            [co.nclk.linen.core :as linen]
            ;[co.nclk.linen.connector.http :refer [connector]]
            [co.nclk.linen.connector.handler :refer [connector]]
            [co.nclk.laundry.models.common :as models]
            ))

(def dbconfig models/config)

(def laundry-config
  (when-let [laundry-file (clojure.java.io/resource "laundry.yaml")]
    (-> laundry-file slurp yaml/parse-string)))

(def testrun-map
  (atom {}))

(defn add-test! [id config]
  (swap! testrun-map #(assoc % id config)))
(defn remove-test! [id]
  (swap! testrun-map #(dissoc % id)))


(defn test-status!
  [status test-run-id]
  (j/with-db-transaction [conn dbconfig]
    (j/update! conn :test_run
      {:status status}
      ["id = ?" test-run-id])))


;;;; Channels

(def job-queue (a/chan))

(defn do-job
  [{:keys [job test-run-id config]}]
  (swap! testrun-map #(assoc % test-run-id config))
  (test-status! "running" test-run-id)
  (job)
  (remove-test! test-run-id))

(def thread-pool
  ;; defaults to 1
  (doseq [t (range 0 (-> laundry-config (:run-concurrency 1)))]
    (a/thread
      (loop []
        (when-let [job-config (a/<!! job-queue)]
          (do-job job-config)
          (recur))))))

;;;;

(defn handlerfn
  [r]
  (fn [s]
    (let [rs (j/with-db-transaction [conn dbconfig]
               (j/query conn
                 [(format "select * from %s where name=?" r)
                  s]))]
      (when-not (empty? rs)
        (-> rs first :data)))))

(def ctor
  (connector
    (handlerfn "program")
    (handlerfn "module")))

(defn pret
  [x]
  (println x)
  x)

(defn merge-evaluated-configs
  [configs config]
  (if (empty? configs)
    config
    (recur
      (drop 1 configs)
      (let [env (:env config)
            arg (first configs)]
        (assoc config :env
                      (merge env
                             (linen/evaluate
                               {(-> arg :name) (-> arg :data)}
                               config)))))))


(defn get-logger
  [test-run-id]
  (reify clojure.tools.logging.impl.Logger
    (enabled? [self level] true)
    (write! [self level throwable message]
      (j/with-db-transaction [conn dbconfig]
        (if (= level :checkpoint)
          (j/insert! conn :checkpoint
            {:id (-> message :runid)
             :test_run_id test-run-id
             :success (-> message :success :value true?)
             :data message})
          (if throwable
            ;; FIXME not sure what "throwable" means
            ;; in this context but we never get here:
            (j/insert! conn :log_entry
              {:test_run_id test-run-id
               :level (name level)
               :message (str "throwable: " message)})
            (j/insert! conn :log_entry
              {:test_run_id test-run-id
               :level (name level)
               :message (or message "[empty]")})))))))


(defn get-logger-factory
  [test-run-id]
  (let [logger (get-logger test-run-id)]
    (reify clojure.tools.logging.impl.LoggerFactory
      (get-logger [self namesp] logger)
      (name [self] "dblogger"))))


(defn log-run-exception!
  [e]
  (when-not (nil? e)
    (log :fatal (.getMessage e))
    (log :trace
         (with-out-str
           (clojure.pprint/pprint
             (.getStackTrace e))))
    (recur (.getCause e))))


(defn do-run
  [config]
  (try (linen/run config)
    (catch Exception e
      (log-run-exception! e))))


(defn harvest!
  [config return test-run-id]
  (doseq [hkey (-> config :harvest)]
    (when-let [data (linen/harvest return hkey)]
      (j/with-db-transaction [conn dbconfig]
        (j/insert! conn :harvest
          {:name hkey
           :test_run_id test-run-id
           :data data})))))


(defn checkpoints!
  [return test-run-id config]
  (let [checkpoints (linen/returns return)]
    (j/with-db-transaction [conn dbconfig]
      (j/update! conn :test_run
        {:status (if @(:runnable? config) "complete" "interrupted")
         :num_checkpoints (count checkpoints)
         :num_failures (->> checkpoints
                         (filter
                           #(-> % :success :value false?))
                         count)}
        ["id = ?" test-run-id]))))


(defn raw-result!
  [return test-run-id]
  (j/with-db-transaction [conn dbconfig]
    (j/insert! conn :raw_result
      {:test_run_id test-run-id
       :raw return})))


(defn test-run!
  [test-run-id seed program-name env]
  (j/with-db-transaction [conn dbconfig]
    (j/insert! conn :test_run
      {:id test-run-id
       :seed seed
       :program_name program-name
       :env env})))


(defn run-with-logger
  [test-run-id config]
  (try
    (let [lfactory (get-logger-factory test-run-id)]
      (binding [*logger-factory* lfactory]
        (let [return (do-run config)]
          (harvest! config return test-run-id)
          (checkpoints! return test-run-id config)
          (raw-result! return test-run-id)
          nil)))
    (catch Exception ie
      (test-status! "error" test-run-id))
    (finally (remove-test! (str test-run-id)))))


(defn default-config
  [seed hkeys program]
  {:env {}
   :runnable? (atom true)
   :thread (atom nil)
   :effective (long seed)
   :harvest (or hkeys [])
   :log-checkpoints? true
   :data-connector ctor
   :genv (atom
           (into {}
             (for [[k v] (System/getenv)]
               [(keyword k) v])))
   :merge-global-environment false
   :program (:data program)})


(defn run [program configs & [hkeys seed]]
  (let [seed (or seed (.getTime (java.util.Date.)))
        config (merge-evaluated-configs
                 configs
                 (merge (default-config seed hkeys program) laundry-config))
        test-run-id (java.util.UUID/randomUUID)]

    (println "Seed:" seed)
    (println "Evaluated environment:")
    (clojure.pprint/pprint config)

    (test-run! test-run-id seed (:name program) (:env config))

    (test-status! "pending" test-run-id)
    (a/go (a/>! job-queue {:job #(run-with-logger test-run-id config)
                         :config config
                         :test-run-id test-run-id}))
    test-run-id))

(defn interrupt
  [testrun-id]
  (when-let [config (-> @testrun-map (get (java.util.UUID/fromString testrun-id)))]
    (swap! (:runnable? config) (fn [_] false))))

(defn kill
  [testrun-id]
  (when-let [thread @(-> @testrun-map (get testrun-id) :thread)]
    (.stop thread)
    (test-status! "killed" testrun-id))
    (remove-test! testrun-id))

