(ns formic.field
  (:require [formic.validation :as fv]
            [struct.core :as st]
            [cljs.pprint :refer [pprint]]
            [reagent.core :as r]))

;; touch all
;; --------------------------------------------------------------

(defn touch-all! [fields]
  (doseq [f fields]
    (cond
      (or (:compound f)
          (:flex f))
      (touch-all! @(:current-value f))
      :else
      (reset! (:touched f) true))))

;; serialization
;; --------------------------------------------------------------

(declare serialize)

(defn serialize-field [f]
  (cond
    (:compound f)
    (assoc ((:serializer f)
            (serialize @(:current-value f)))
           :_compound (:compound f))
    ;; flexible
    (:flex f)
    (mapv (fn [flex-f]
             (serialize-field flex-f))
          @(:current-value f))
    @(:touched f)
    (let [field-type (:type f)
          serializer (:serializer f)]
      (serializer @(:current-value f)))
    :else nil))

(defn serialize
  ([fields] (serialize {} fields))
  ([result fields]
   (if (nil? fields) result
       (reduce (fn [result field]
                 (if-let  [serialized (serialize-field field)]
                   (assoc result (:id field) serialized)
                   result))
               result fields))))

;; error handling
;; --------------------------------------------------------------

(defn validate-field [validation touched current-value]
  (when (and @touched validation)
    (first (st/validate-single
            @current-value
            validation))))

(defn validate-compound [form-schema f]
  (when (:validation f)
   (first
    (st/validate
     @(:current-value f)
     (:validation f)))))

(defn validate-all [fields]
  (loop [fields fields acc {}]
    (if (empty? fields) (not-empty acc)
     (let [f (first fields)]
       (if-let [err (not-empty
                     (cond
                       (:flex f)
                       (validate-all @(:current-value f) {})
                       (:compound f)
                       (validate-all @(:current-value f) {})
                       :else @(:err f)))]
         (recur (rest fields) (assoc acc (:id f) err))
         (recur (rest fields) acc))))))

;; field prep
;; --------------------------------------------------------------

(declare prepare-field)

(defn prepare-field-basic [{:keys [values serializers parsers]} path f]
  "Prepares a single basic field type."
  (let [original-value (or (get-in values path)
                           (:default f)
                           (when (and (or (= type :select)
                                          (= type :radios))
                                      (:options f))
                             (first (first (:options f)))))
        parser (or (:parser f)
                   (get parsers (:type f) identity))
        serializer (or (:serializer f)
                       (get serializers (:type f) identity))
        parsed-value (parser original-value)
        current-value (r/atom parsed-value)
        touched (r/atom (not (nil? parsed-value)))
        err (r/track validate-field (:validation f) touched current-value)]
     (assoc f
            :original-value parsed-value
            :current-value current-value
            :serializer serializer
            :err err
            :touched touched)))

(defn prepare-field-compound [{:keys [values compound serializers] :as form-schema}
                              path
                              f]
  (let [compound-type (:compound f)
        compound-schema (get compound compound-type)
        compound-fields (:fields compound-schema)
        compound-serializer (or (:serializer f)
                                (get serializers compound-type identity))
        prepared-fields
        (mapv
         (fn [field]
           (let [new-path (conj path (:id field))]
             (prepare-field form-schema new-path field)))
         compound-fields)]
    (assoc f
           :serializer compound-serializer
           :validation (:validation compound-schema)
           :current-value (r/atom prepared-fields))))

(defn prepare-field-flexible [{:keys [values] :as form-schema} path f]
  (let [flex-values (get-in values path [])]
    (assoc f
           :next (atom (count flex-values))
           :current-value
           (r/atom
            (vec
             (map-indexed
              (fn [n v]
                (let [field-type (keyword (:_compound v))
                      new-field-name (str (name (:id f)) "-" n "-" (name field-type))
                      new-path (conj path n)]
                  (prepare-field form-schema
                                 (conj path n)
                                 {:id new-field-name
                                  :title field-type
                                  :compound field-type
                                  :path new-path})))
              (get-in values path flex-values)))))))

(defn prepare-field [form-schema path f]
  (cond
    (:compound f)
    (prepare-field-compound form-schema path f)
    (:flex f)
    (prepare-field-flexible form-schema path f)
    :else
    (prepare-field-basic form-schema path f)))

(defn prepare-all-fields [form-schema]
  (doall
   (mapv
    (fn [f]
      (prepare-field form-schema [(:id f)] f))
    (:fields form-schema))))

;; flex
;; --------------------------------------------------------------

(defn add-field [form-schema f field-type]
  (let [next @(:next f)
        new-field-name (str (name (:id f)) "-" next "-" (name field-type))
        new-field (prepare-field-compound
                   form-schema
                   nil
                   {:id new-field-name
                    :title field-type
                    :compound field-type})]
    (swap! (:current-value f) conj new-field)
    (swap! (:next f) inc)))
