(ns formic.frontend
  (:require [formic.components.inputs :as formic-inputs]
            [formic.util :as formic-util]
            [formic.field :as formic-field]
            [reagent.core :as r]
            [cljs.pprint :refer [pprint]]
            [cljsjs.react-flip-move]
            [clojure.string :as s]))

(def flip-move (r/adapt-react-class js/FlipMove))

(declare field)

(defn get-field [form-schema path]
  (get-in (:fields form-schema) path))

(defn compound-field [form-schema form-state f path]
  (let [current-value (r/cursor form-state path)
        compound-schema (get-in form-schema [:compound (:compound f)])
        compound-error
        (formic-field/validate-compound
                        path
                        (:validation compound-schema)
                        current-value)]
    [:fieldset.formic-compound
     {:class (str (name (:compound f)))}
     [:h4.formic-compound-title
      (or (:title f) (formic-util/format-kw (:id f)))]
     [:ul.formic-compound-fields
      (doall
       (for [cf (:fields compound-schema)]
         ^{:key [(or (:_key f) (:id f)) (:id cf)]}
         [:li
          [field form-schema form-state
           (assoc cf :_key (:_key f))
           (conj path (:id cf))]]))]
     (when compound-error
       [:ul.compound-errors
        (for [[id err] compound-error]
          ^{:key id}
          [:li
           [:h4.error
            [:strong (formic-util/format-kw id)] ": " err]])])]))

(defn flexible-controls [current-value n]
  (let [is-first (= n 0)
        is-last (= (-> current-value deref count dec) n)]
    [:ul.formic-flex-controls
     [:li.up
      [:a
       {:href "#"
        :class (when is-first "disabled")
        :on-click
        (fn [ev]
          (.preventDefault ev)
          (when-not is-first
            (swap! current-value formic-util/vswap n (dec n))))}
       "↑"]]
     [:li.down
      [:a
       {:href "#"
        :class (when is-last "disabled")
        :on-click
        (fn [ev]
          (.preventDefault ev)
          (when-not is-last
            (swap! current-value formic-util/vswap n (inc n))))}
       "↓"]]
     [:li.delete
      [:a {:href "#"
           :on-click
           (fn [ev]
             (.preventDefault ev)
             (swap! current-value formic-util/vremove n))}
       "✗"]]]))

(enable-console-print!)

(defn flexible-field [{:keys [fields compound]
                       :as form-schema}
                      form-state f path]
  (let [current-value (r/cursor form-state path)
        next (r/atom (or (count @current-value) 0))]
    (fn [form-schema form-state f path]
      (let [current-value (r/cursor form-state path)]
        [:fieldset.formic-flex
         [:h4.formic-flex-title (formic-util/format-kw (:id f))]
         [flip-move {:enter-animation "fade"
                     :leave-animation "fade"}
          (doall
           (for [n (range (count @current-value))
                 :let [ff (get @current-value n)]]
             ^{:key (:_key ff)}
             [:div.formic-flex-field
              [flexible-controls current-value n]
              [field form-schema form-state ff (conj path n)]]))]
         [:ul.formic-flex-add
          (for [field-type (:flex f)]
            ^{:key field-type}
            [:li
             [:a.button
              {:href "#"
               :on-click (fn [ev]
                           (.preventDefault ev)
                           (formic-field/add-field form-schema
                                                   next
                                                   f
                                                   current-value
                                                   field-type))}
              [:span.plus "+"] (formic-util/format-kw field-type)]])]]))))

(defn formic-buttons
  "Renders the buttons for a set of formic fields.
  Each button has
  - :id - id for the button
  - :label - label for the button (optional - defaults to formatted id)
  - :on-click - Action to perform on click.
              - calls preventDefault on event
              - fn receives current form-state _atom_
  "
  [buttons form-state]
  [:div.formic-buttons
   [:ul
    (for [b buttons]
      (when b
        (let [id (formic-util/format-kw (:id b))]
          ^{:key b}
          [:li
           [:button.formic-buttons-button
            {:name     id
             :id       id
             :on-click (fn [ev]
                         (.preventDefault ev)
                         ((:on-click b) form-state))}
            (or (:label b) (s/capitalize id))]])))]])

(defn unknown-field [f]
  [:h4 "Unknown:"
   [:pre (with-out-str
           (pprint f))]])

(defn basic-field [form-schema form-state f path]
  (let [current-value (r/cursor form-state (conj path :value))
        form-component
        (or (get-in form-schema [:components (:type f)])
            (get formic-inputs/default-components (:type f))
            unknown-field)
        touched (r/cursor form-state (conj path :touched))
        err (r/track formic-field/validate-field
                     (:validation f)
                     touched
                     current-value)
        final-f (assoc f
                       :path path
                       :touched touched
                       :current-value current-value
                       :err err)]
    [form-component final-f]))

(defn field [form-schema form-state f path]
  [:div.formic-field
   (cond
     (:flex f)
     [flexible-field form-schema form-state f path]
     (:compound f)
     [compound-field form-schema form-state f path]
     :else
     [basic-field form-schema form-state f path])])

(defn formic-fields [form-schema form-state]
  (fn [form-schema form-state]
    [:div.formic-fields
     (for [n (range (count (:fields form-schema)))
           :let [f (get (:fields form-schema) n)]]
       ^{:key n}
       [field form-schema form-state f [(:id f)]])]))
