(ns codox.utils
  "Miscellaneous utility functions."
  (:require [clojure.string :as str]))

(defn- find-minimum [coll]
  (if (seq coll)
    (apply min coll)))

(defn- find-smallest-indent [text]
  (->> (str/split-lines text)
       (remove str/blank?)
       (map #(re-find #"^\s+" %))
       (map count)
       (find-minimum)))

(defn unindent
  "Unindent a block of text by a specific amount or the smallest common
  indentation size."
  ([text]
     (unindent text (find-smallest-indent text)))
  ([text indent-size]
     (let [re (re-pattern (str "^\\s{0," indent-size "}"))]
       (->> (str/split-lines text)
            (map #(str/replace % re ""))
            (str/join "\n")))))

(defn symbol-set
  "Accepts a single item (or a collection of items), converts them to
  symbols and returns them in set form."
  [x]
  (->> (if (coll? x) x [x])
       (filter identity)
       (map symbol)
       (into #{})))

(defn ns-filter
  "Accepts a sequence of namespaces (generated by
  `codox.reader/read-namespaces`), a sequence of namespaces to keep
  and a sequence of namespaces to drop. The sequence is returned with
  all namespaces in `exclude` and all namespaces NOT in `include`
  removed."
  [ns-seq include exclude]
  (let [has-name? (fn [names] (comp (symbol-set names) :name))
        ns-seq    (remove (has-name? exclude) ns-seq)]
    (if include
      (filter (has-name? include) ns-seq)
      ns-seq)))
