(ns orcl.naive
  (:require [orcl.naive.impl :as impl]
            [orcl.naive.vars :as vars]
            [orcl.utils :as utils]
            [orcl.compiler :as compiler]
            [orcl.naive.lib :as lib])
  (:refer-clojure :exclude [compile]))

(defn prepare-primitive [ast]
  (if (= :var (:node ast))
    (let [s (:source ast)]
      (case (:type s)
        :site {:node  :const
               :value (impl/make-site (:type (:source s)) (:definition s))}
        :refer {:node      :refer
                :namespace (:namespace s)
                :symbol    (:var ast)}
        {:node :var
         :var  (:var ast)}))
    ast))

(defn prelude-const [definition]
  {:node :const
   :value (impl/make-site :prelude definition)})

(defn pattern-binding [p]
  (case (:type p)
    :wildcard "_"
    :var (:var p)))

(declare compile)

(defn compile-def [def]
  {:name   (:name def)
   :locals (:locals def)
   :params (map pattern-binding (:params (first (:instances def))))
   :body   (compile (:body (first (:instances def))))})

(defn compile [ast]
  (case (:node ast)
    (:sequential :pruning) {:node    (:node ast)
                            :binding (pattern-binding (:pattern ast))
                            :left    (compile (:left ast))
                            :right   (compile (:right ast))}
    (:otherwise :parallel) {:node  (:node ast)
                            :left  (compile (:left ast))
                            :right (compile (:right ast))}
    :defs-group (assoc ast
                  :expr (compile (:expr ast))
                  :defs (map compile-def (:defs ast)))
    :call {:node   :call
           :target (prepare-primitive (:target ast))
           :args   (map prepare-primitive (:args ast))}
    (:tuple :list) {:node   :call
                    :target (case (:node ast) :tuple (prelude-const "_MakeTuple") :list (prelude-const "_MakeList"))
                    :args   (map prepare-primitive (:values ast))}
    :record {:node   :call
             :target (prelude-const "_MakeRecord")
             :args   (concat [{:node :const :value (map first (:pairs ast))}]
                             (map (comp prepare-primitive second) (:pairs ast)))}
    :field-access {:node   :call
                   :target (prelude-const "_FieldAccess")
                   :args   [(prepare-primitive (:target ast)) {:node :const :value (:field ast)}]}
    (:stop :const) ast
    :var {:node   :call
          :target (prelude-const "Let")
          :args   [(prepare-primitive ast)]}))

(defn make-snapshot [values prev-coeffects dependencies]
  (let [coeffects (map (fn [[k {:keys [definition]}]] [k definition]) @impl/*coeffects*)
        killed    (keep (fn [[k stack]] (when-not (impl/alive? stack) k)) prev-coeffects)
        state     (into {} (concat (map (fn [[k {:keys [stack]}]] [k stack]) @impl/*coeffects*)
                                   (apply dissoc prev-coeffects killed)))]
    (reify compiler/Snapshot
      (values [_] @values)
      (coeffects [_] coeffects)
      (killed-coeffects [_] killed)
      (unblock [_ coeffect value]
        (when-let [stack (get state coeffect)]
          (reset! values [])
          (binding [impl/*coeffects*        (atom {})
                    impl/*exectution-queue* (atom ())
                    impl/*dependencies*     dependencies]
            (impl/publish stack value)
            (impl/halt stack)
            (impl/execution-loop)
            (make-snapshot values (dissoc state coeffect) dependencies)))))))

(defn eval* [program dependencies]
  (let [values (atom [])]
    (binding [impl/*coeffects*        (atom {})
              impl/*exectution-queue* (atom ())
              impl/*dependencies*     dependencies]
      (impl/execution-loop program (impl/result-frame (fn [v] (swap! values conj v))))
      (make-snapshot values {} dependencies))))

(defn backend []
  (reify compiler/Backend
    (prelude [_] lib/prelude)
    (analyzer-options [_] {:deflate?     true
                           :patterns?    true
                           :clauses?     true
                           :conditional? true})
    (compile [this ast] (compiler/compile this ast {}))
    (compile [_ ast dependenices]
      (let [compiled (compile ast)]
        (reify compiler/Program
          (eval [this] (compiler/eval this {}))
          (eval [_ dependenices]
            (eval* compiled dependenices)))))
    (compile-namespace [this ast] (compiler/compile-namespace this ast {}))
    (compile-namespace [_ decls dependencies]
      (let [defs (map compile-def decls)
            group {:env  {}
                   :defs defs}]
        (into {} (for [{:keys [name] :as d} defs]
                   [name (impl/make-closure group d)]))))))
