(ns orcl.fs
  #?@(:clj
      [(:require [clojure.java.io :as io])
       (:import [java.net URI]
                [java.nio.file Paths Files FileVisitOption LinkOption FileSystems FileSystemNotFoundException]
                [java.io FileNotFoundException])]))

(defprotocol FileSystem
  (read-file [this file]))

#?(:clj
   (do
     (def resources-file-system
       (reify FileSystem
         (read-file [this file]
           (let [r (io/resource file)]
             (if r
               (slurp r)
               (throw (ex-info (str "File not found " file) {:file file})))))))

     (defn filesystem [prefix]
       (reify FileSystem
         (read-file [this file] (slurp (io/file prefix file)))))

     (defn preread-prefix [prefix]
       (let [prefix-uri (.toURI (io/file prefix))]
         (into {}
               (for [f (file-seq (io/file prefix))
                     :when (.isFile f)
                     :let [uri (.toURI f)]]
                 [(str (.relativize prefix-uri uri)) (slurp f)]))))

     (defn uri->path [uri]
       (try
         (Paths/get uri)
         (catch FileSystemNotFoundException e
           (FileSystems/newFileSystem uri {"create" "ture"})
           (uri->path uri))))

     (defn preread-resources [prefix]
       (let [prefix-uri (.toURI (io/resource prefix))
             path       (uri->path prefix-uri)
             stream     (Files/walk path (into-array FileVisitOption []))]
         (->> stream
              (.iterator)
              (iterator-seq)
              (filter #(not (Files/isDirectory % (into-array LinkOption []))))
              (map (fn [p]
                     [(str (io/file prefix (str (.relativize path p))))
                            (slurp (.toUri p))]))
              (into {}))))

     (defmacro preread-fs [prefix]
       `(let [m# ~(preread-prefix prefix)]
          (reify FileSystem
            (read-file [this# file#] (get m# file#)))))

     (defmacro preread-resources-fs [prefix]
       `(let [m# ~(preread-resources prefix)]
          (reify FileSystem
            (read-file [this# file#] (get m# file#)))))))

(defn in-memory-file-system
  [m]
  (reify FileSystem
    (read-file [this file] (m file))))

(defn multiple-fs [filesystems]
  (reify FileSystem
    (read-file [_ file]
      (or (some #(try (read-file % file)
                      (catch #?(:clj Exception :cljs js/Error) e
                        nil))
                filesystems)
          (throw (ex-info (str "File not found " file) {:file file}))))))


