(ns orcl.testkit.core
  (:require [orcl.testkit.tests :as tests]
            [orcl.testkit.proto :as proto]
            [orcl.parser :as parser]
            [orcl.analyzer :as analyzer])
  (:gen-class))

(defn normalize-desc [x]
  (cond
    (map? x) x
    (vector? x) {:expectations (mapv (fn [v] {:type :basic :value v}) x)}
    (set? x) {:expectations [{:type :permutable :values x}]}
    :else {:expectations [{:type :basic :value x}]}))

(def tests (select-keys tests/tests [:arithmetic]))

(def ^:dynamic *compiler*)

(defn run-and-check [program test-desc]
  (let [values
             (loop [{:keys [values state coeffects]} (proto/run program)]
               (if (seq coeffects)
                 (let [[id definition] (first coeffects)
                       v          (or (get (:blocks test-desc) definition)
                                      (throw (ex-info "Unexpected coeffect" {:definition definition})))]
                   (recur (proto/unblock program state id v)))
                 values))]
    (loop [[e & expectations] (:expectations test-desc) values values]
      (cond
        (and (nil? e) (empty? values)) :ok
        (and e (empty? values)) (throw (ex-info "Value expected" {:expectation e}))
        (and (nil? e) (seq values)) (throw (ex-info "Unexpected value" {:values values}))
        :else (case (:type e)
                :basic (if (= (:value e) (first values))
                         (recur expectations (rest values))
                         (throw (ex-info "Unexpected value" {:value       (first values)
                                                             :expectation e})))
                :permutable (if (contains? (:values e) (first values))
                              (if (= 1 (count (:values e)))
                                (recur expectations (rest values))
                                (recur (cons (update e :values disj (first values)) expectations) (rest values)))
                              (throw (ex-info "Unexpected value" {:value       (first values)
                                                                  :expectation e}))))))))

(defn -main [compiler & args]
  (let [compiler-sym (symbol compiler)]
    (require (symbol (namespace compiler-sym)))
    (binding [*compiler* ((resolve compiler-sym))]
      (let [prelude (proto/prelude *compiler*)]
        (doseq [[suite tests] tests]
          (prn (str suite))
          (doseq [[i [t desc]] (map-indexed vector tests)]
            (let [parsed   (parser/parse t)
                  analyzed (analyzer/analyze parsed prelude)
                  program  (proto/compile *compiler* analyzed)]
              (prn "---T" t)
              (try
                (run-and-check program (normalize-desc desc))
                (catch clojure.lang.ExceptionInfo e
                  (prn e) (throw e))))))))))