(ns orcl.testkit.core
  (:require [orcl.testkit.tests :as tests]
            [orcl.testkit.proto :as proto]
            [orcl.parser :as parser]
            [orcl.analyzer :as analyzer])
  (:gen-class))


(defn normalize-desc [x]
  (cond
    (map? x) (update x :coeffects (fn [coeffects] (map #(update % :expectations normalize-desc) coeffects)))
    (vector? x) {:expectations (mapv (fn [v] {:type :basic :value v}) x)}
    (set? x) {:expectations [{:type :permutable :values x}]}
    :else {:expectations [{:type :basic :value x}]}))

(defn normalize-values-spec [spec]
  (if (or (vector? spec) (set? spec) (map? spec))
    spec
    [spec]))

(defn normalize-spec [spec]
  (if (map? spec)
    (update spec :values normalize-values-spec)
    {:values (normalize-values-spec spec)}))

(defn coeffect-id-by-definition [pending-coeffects definition]
  (some (fn [[id d]] (when (= d definition) id)) pending-coeffects))

(defn check-res [res spec pending-coeffects]
  (let [spec (normalize-spec spec)]
    (let [failed (cond
                   (set? (:values spec))
                   (or (not= (count (:values spec)) (count (:values res)))
                                            (not= (:values spec) (set (:values res))))

                   (and (map? (:values spec)) (::tests/one-of (:values spec)))
                   (or (not= 1 (count (:values res)))
                       (not (contains? (::tests/one-of (:values spec)) (first (:values res)))))

                   :else (not= (:values spec) (:values res)))]
      (when failed
        (throw (ex-info "Unexpected values" {:expected (:values spec) :actual (:values res)}))))
    (let [expected-killed (map (partial coeffect-id-by-definition pending-coeffects) (:killed-coeffects spec))]
      (when (not= (set expected-killed) (set (:killed-coeffects res)))
        (throw (ex-info "Unexpected killed-coeffects" {:expected expected-killed :actual (:killed-coeffects res)}))))))

(defn run-and-check [compiler test]
  (let [[program run-spec & unblock-specs] test
        _        (prn "---T" program)
        parsed   (parser/parse program)
        compiled (proto/compile compiler parsed)]
    (loop [res               (proto/run compiled)
           spec              {:values run-spec}
           [[coeffect-definition realized-value next-spec] & unblock-specs] unblock-specs
           pending-coeffects {}]
      (check-res res spec pending-coeffects)
      (let [pending-coeffects (apply dissoc (merge pending-coeffects (into {} (:coeffects res)))
                                     (:killed-coeffects res))]
        (cond
          (and coeffect-definition (empty? pending-coeffects))
          (throw (ex-info "Expected coeffect" {:coeffect coeffect-definition}))

          coeffect-definition
          (let [coeffect-id (coeffect-id-by-definition pending-coeffects coeffect-definition)]
            (recur (proto/unblock compiled (:state res) coeffect-id realized-value)
                   next-spec
                   unblock-specs
                   (dissoc pending-coeffects coeffect-id)))

          (seq pending-coeffects)
          (throw (ex-info "Pending coeffects" {:pending-coeffects pending-coeffects})))))))

(defn -main [compiler & args]
  (let [compiler-sym (symbol compiler)
        _            (require (symbol (namespace compiler-sym)))
        compiler     ((resolve compiler-sym))]
    (doseq [[suite tests] tests/tests]
      (prn (str suite))
      (doseq [t tests]
        (try
          (run-and-check compiler t)
          (catch clojure.lang.ExceptionInfo e
            (prn e) (throw e)))))))