(ns orcl.testkit.core
  (:require [orcl.testkit.tests :as tests]
            [orcl.testkit.proto :as proto]
            [orcl.parser :as parser]
            [orcl.analyzer :as analyzer]
            [criterium.core :as criterium]
            [capacitor.core :as capacitor])
  (:refer-clojure :exclude [compile])
  (:import [java.net URI])
  (:gen-class))


(defn normalize-desc [x]
  (cond
    (map? x) (update x :coeffects (fn [coeffects] (map #(update % :expectations normalize-desc) coeffects)))
    (vector? x) {:expectations (mapv (fn [v] {:type :basic :value v}) x)}
    (set? x) {:expectations [{:type :permutable :values x}]}
    :else {:expectations [{:type :basic :value x}]}))

(defn normalize-values-spec [spec]
  (if (or (vector? spec) (set? spec) (map? spec))
    spec
    [spec]))

(defn normalize-spec [spec]
  (if (map? spec)
    (update spec :values normalize-values-spec)
    {:values (normalize-values-spec spec)}))

(defn coeffect-id-by-definition [pending-coeffects definition]
  (some (fn [[id d]] (when (= d definition) id)) pending-coeffects))

(defn check-res [res spec pending-coeffects]
  (let [spec (normalize-spec spec)]
    (let [failed (cond
                   (set? (:values spec))
                   (or (not= (count (:values spec)) (count (:values res)))
                       (not= (:values spec) (set (:values res))))

                   (and (map? (:values spec)) (::tests/one-of (:values spec)))
                   (or (not= 1 (count (:values res)))
                       (not (contains? (::tests/one-of (:values spec)) (first (:values res)))))

                   :else (not= (:values spec) (:values res)))]
      (when failed
        (throw (ex-info "Unexpected values" {:expected (:values spec) :actual (:values res)}))))
    (let [expected-killed (map (partial coeffect-id-by-definition pending-coeffects) (:killed-coeffects spec))]
      (when (not= (set expected-killed) (set (:killed-coeffects res)))
        (throw (ex-info "Unexpected killed-coeffects" {:expected expected-killed :actual (:killed-coeffects res)}))))))

(defn run-and-check [compiled [_ run-spec & unblock-specs]]
  (loop [res               (proto/run compiled)
         spec              {:values run-spec}
         [[coeffect-definition realized-value next-spec] & unblock-specs] unblock-specs
         pending-coeffects {}]
    (check-res res spec pending-coeffects)
    (let [pending-coeffects (apply dissoc (merge pending-coeffects (into {} (:coeffects res)))
                                   (:killed-coeffects res))]
      (cond
        (and coeffect-definition (empty? pending-coeffects))
        (throw (ex-info "Expected coeffect" {:coeffect coeffect-definition}))

        coeffect-definition
        (let [coeffect-id (coeffect-id-by-definition pending-coeffects coeffect-definition)]
          (recur (proto/unblock compiled (:state res) coeffect-id realized-value)
                 next-spec
                 unblock-specs
                 (dissoc pending-coeffects coeffect-id)))

        (seq pending-coeffects)
        (throw (ex-info "Pending coeffects" {:pending-coeffects pending-coeffects}))))))

(defn compile [compiler test]
  (let [[program] test
        ;_        (prn "---T" program)
        parsed   (parser/parse program)]
    (proto/compile compiler parsed)))

(defn run-test [compiler t]
  (run-and-check (compile compiler t) t))

(defn run-all-tests [compiler]
  (doseq [[suite tests] tests/tests]
    (doseq [[program :as t] tests]
      (try
        (prn program)
        (run-test compiler t)
        (catch clojure.lang.ExceptionInfo e
          (prn e) (throw e))))))

(defn required-env [n]
  (or (System/getenv n)
      (do
        (prn (format "Required ENV variable %s" n))
        (System/exit 1))))

(defn benchmarks [compiler tags]
  (let [addr   (URI. (required-env "INFLUX_CONNECT"))
        influx (capacitor/make-client {:host     (.getHost addr)
                                       :scheme   (.getScheme addr)
                                       :port     (.getPort addr)
                                       :username (System/getenv "INFLUX_USER")
                                       :password (System/getenv "INFLUX_PASS")
                                       :db       (required-env "INFLUX_DB")})
        results (for [[suite tests] tests/tests
                        :let [compiled (map (partial compile compiler) tests)
                              res      (criterium/quick-benchmark
                                         (doseq [[compiled t] (map vector compiled tests)]
                                           (run-and-check compiled t))
                                         {})
                              [mean] (:mean res)]]
                     [suite (:mean res)])
        now (System/currentTimeMillis)
        tags (into {} (map (fn [t] (let [[_ tag value] (re-find #"(.+)=(.+)" t)]
                                     (when-not tag
                                       (prn (format "Bad formatted tag %s" t))
                                       (System/exit 1))
                                     [tag value]))))]
    (capacitor/write-points influx
                            (for [[suite mean] results]
                              {:measurement (name suite)
                               :tags tags
                               :fields      {"value" mean}
                               :timestamp   now}))))

(defn -main [compiler & [mode & args]]
  (let [compiler-sym (symbol compiler)
        _            (require (symbol (namespace compiler-sym)))
        compiler     ((resolve compiler-sym))]
    (case mode
      "benchmarks" (benchmarks compiler args)
      (run-all-tests compiler))))