;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns distantia.core
  (:require [utilis.map :refer [key-paths]])
  #?(:clj (:import [distantia NumericalArray])))

(declare map-diff map-patch vec-diff vec-patch)

(defn diff
  [a b & {:keys [shallow]}]
  (cond
    (and (map? a) (map? b)) (map-diff a b :shallow shallow)
    (and (vector? a) (vector? b)) (vec-diff a b)
    :else [:s b]))

(defn patch
  [a [type p :as patch]]
  (cond
    (and (map? a) (= :m type)) (map-patch a patch)
    (and (vector? a) (= :v type)) (vec-patch a patch)
    (= :s type) p))


;;; Private

(defn- map-diff
  [x y & {:keys [shallow]}]
  (let [prefix? (fn [pre s] (= (take (count pre) s) pre))]
    [:m (loop [a (transient []) r (transient []) c (transient [])
               paths (distinct (concat (key-paths x) (key-paths y)))]
          (if-let [kp (first paths)]
            (let [xv (get-in x kp ::not-found)
                  yv (get-in y kp ::not-found)
                  xv? (not= ::not-found xv)
                  yv? (not= ::not-found yv)]
              (cond
                (and xv? yv? (= xv yv))
                (recur a r c (next paths))

                (and xv? yv? (vector? xv) (vector? yv))
                (recur a r (conj! c [kp (vec-diff xv yv) :s])
                       (remove (partial prefix? kp) (next paths)))

                (and xv? yv? (map? xv) (map? yv) (not shallow))
                (recur a r c (next paths))

                (and xv? yv?)
                (recur a r (conj! c [kp yv])
                       (remove (partial prefix? kp) (next paths)))

                xv?
                (recur a (conj! r kp) c
                       (remove (partial prefix? kp) (next paths)))

                yv? (recur (conj! a [kp yv]) r c
                           (remove (partial prefix? kp) (next paths)))))
            (cond-> {}
              (pos? (count a)) (assoc :a (persistent! a))
              (pos? (count r)) (assoc :r (persistent! r))
              (pos? (count c)) (assoc :c (persistent! c)))))]))

(defn- map-patch
  [m [_patch-type patch]]
  (as-> m $
    (reduce (fn [m r]
              (if (= 1 (count r))
                (dissoc m (first r))
                (cond-> m
                  (not= ::not-found (get-in m (butlast r) ::not-found))
                  (update-in (butlast r) dissoc (last r))))) $ (:r patch))
    (reduce (fn [m c]
              (if (= 2 (count c))
                (apply assoc-in m c)
                (let [[key-path patch _] c]
                  (update-in m key-path vec-patch patch)))) $ (:c patch))
    (reduce (fn [m a] (apply assoc-in m a)) $ (:a patch))))

(declare lcs-matrix lcs-diff)

(defn- vec-diff
  [v w]
  [:v (lcs-diff (lcs-matrix v w) v w)])

(defn- vec-patch
  [v [_patch-type patch]]
  (vec
   (:output
    (reduce (fn [{:keys [input _output] :as v} [op count values]]
              (case op
                :k (-> v (update :output concat (take count input))
                       (update :input (partial drop count)))
                :r (-> v (update :input (partial drop count)))
                :a (-> v (update :output concat values))))
            {:input v :output []} patch))))

(def ^:private wrap-aset #?(:clj (fn [a i j v] (NumericalArray/set_long_array_2d a i j v))
                            :cljs aset))
(def ^:private wrap-aget #?(:clj (fn [a i j] (NumericalArray/get_long_array_2d a i j))
                            :cljs aget))

(defn- lcs-matrix
  [s t & {:keys [compare] :or {compare =}}]
  (let [rows (inc (count s))
        cols (inc (count t))
        a #?(:clj  (NumericalArray/long_array_2d rows cols)
             :cljs (make-array Long rows cols))]
    (doseq [r (range rows)] (wrap-aset a r 0 0))
    (doseq [c (range cols)] (wrap-aset a 0 c 0))
    (doseq [i (range 1 rows)
            j (range 1 cols)]
      (if (compare (nth s (dec i)) (nth t (dec j)))
        (wrap-aset a i j (inc (wrap-aget a (dec i) (dec j))))
        (wrap-aset a i j (max (wrap-aget a i (dec j))
                              (wrap-aget a (dec i) j)))))
    a))

(defn- lcs-diff
  [lcs-matrix s t & {:keys [compare] :or {compare =}}]
  (let [m lcs-matrix]
    (loop [p '() i (count s) j (count t)]
      (let [[op count values] (first p)]
        (cond (and (< 0 i) (< 0 j) (compare (nth s (dec i)) (nth t (dec j))))
              (recur (if (= :k op)
                       (cons [:k (inc count)] (rest p))
                       (cons [:k 1] p))
                     (dec i) (dec j))
              (and (< 0 j) (or (= i 0) (<= (wrap-aget m (dec i) j) (wrap-aget m i (dec j)))))
              (recur (if (= :a op)
                       (cons [:a (inc count) (cons (nth t (dec j)) values)] (rest p))
                       (cons [:a 1 [(nth t (dec j))]] p))
                     i (dec j))
              (and (< 0 i) (or (= j 0) (>  (wrap-aget m (dec i) j) (wrap-aget m i (dec j)))))
              (recur (if (= :r op)
                       (cons [:r (inc count)] (rest p))
                       (cons [:r 1] p))
                     (dec i) j)
              :else p)))))
