(ns fides.certificates
  (:require [clojure.string :refer [join]]
            [tempus.core :as t]
            [tempus.interval :as i]
            [tempus.duration :as d]
            [fides.util.bytes :as bytes])
  (:import [java.io File ByteArrayInputStream FileOutputStream]
           [java.util Date]
           [java.time ZoneOffset]
           [java.math BigInteger]
           [java.security KeyFactory KeyPair KeyPairGenerator PrivateKey PublicKey SecureRandom]
           [java.security.cert CertificateFactory]
           [java.security.spec ECGenParameterSpec PKCS8EncodedKeySpec]
           [sun.security.x509 AlgorithmId CertificateAlgorithmId X509CertImpl X509CertInfo X509CRLEntryImpl X509CRLImpl
            CertificateSerialNumber CertificateSubjectName CertificateValidity CertificateVersion CertificateX509Key SerialNumber X500Name
            CertificateExtensions BasicConstraintsExtension ExtendedKeyUsageExtension KeyUsageExtension SubjectAlternativeNameExtension
            GeneralNames GeneralName DNSName IPAddressName OIDName RFC822Name URIName]
           [sun.security.pkcs10 PKCS10]
           [sun.security.util ObjectIdentifier]
           ))

(defonce ^:private ^String default-key-algorithm "EC")
(defonce ^:private ^String default-signature-algorithm "SHA256withECdSA")
(defonce ^:private ^String defaul-group-name "secp384r1")

(declare ->tempus ->pem
         ^X509CertImpl pem->x509-cert ^PKCS10 pem->pkcs10-csr ^PrivateKey pem->key ^PublicKey pem->public-key ^X509CRLImpl pem->x509-crl
         ^X509CertImpl unsigned-certificate
         pem-bytes
         key-tuple
         ^CertificateExtensions certificate-extensions
         ^X509CertInfo x509-cert-info)


(defn self-signed-certificate
  ([cn validity] (self-signed-certificate cn validity nil))
  ([cn validity extensions]
   (let [[^PublicKey public-key ^PrivateKey key] (key-tuple default-key-algorithm defaul-group-name)
         x500-name (X500Name. (str "CN=" cn))
         x509-cert (doto ^X509CertImpl (unsigned-certificate x500-name public-key x500-name validity extensions)
                     (.sign key default-signature-algorithm))]
     {:certificate (->pem x509-cert)
      :key (->pem key)})))

(defn certificate-signing-request
  [cn]
  (let [[^PublicKey public-key ^PrivateKey key] (key-tuple default-key-algorithm defaul-group-name)
        x500-name (X500Name. (str "CN=" cn))
        pkcs10 (doto (PKCS10. public-key)
                 (.encodeAndSign x500-name key default-signature-algorithm))]
    {:csr (->pem pkcs10)
     :key (->pem key)}))

(defn signed-certificate
  [csr validity extensions signing-cert signing-key]
  (let [pkcs10 (pem->pkcs10-csr csr)
        signing-x509 (pem->x509-cert signing-cert)
        signing-x509-info (x509-cert-info signing-x509)
        signing-PK (pem->key signing-key)
        cert (doto (unsigned-certificate (.get signing-x509-info (str X509CertInfo/SUBJECT "." CertificateSubjectName/DN_NAME))
                                         (.getSubjectPublicKeyInfo pkcs10)
                                         (.getSubjectName pkcs10)
                                         validity
                                         extensions)
               (.sign signing-PK default-signature-algorithm))]
    {:certificate (->pem cert)}))

(defn certificate-revocation-list
  [validity issuer-crt issuer-key crt-coll]
  (let [issuing-pk (pem->key issuer-key)
        issuer-cert-info ^X509CertInfo (-> issuer-crt pem->x509-cert x509-cert-info)]
    {:crl (->pem (doto (X509CRLImpl. (.get issuer-cert-info (str X509CertInfo/SUBJECT "." CertificateSubjectName/DN_NAME))
                                     (Date.)
                                     (Date. (long (t/into :long (t/+ (t/now) (d/days validity)))))
                                     (into-array X509CRLEntryImpl
                                                 (mapv (fn [c]
                                                         (X509CRLEntryImpl. (-> c
                                                                                pem->x509-cert
                                                                                ^X509CertInfo x509-cert-info
                                                                                ^CertificateSerialNumber (.get X509CertInfo/SERIAL_NUMBER)
                                                                                ^SerialNumber (.get CertificateSerialNumber/NUMBER)
                                                                                .getNumber)
                                                                            (Date.))) crt-coll)))
                   (.sign issuing-pk default-signature-algorithm)))}))

(defn certificate-revoked?
  [crl cert]
  (.isRevoked (pem->x509-crl crl) (pem->x509-cert cert)))

(defn not-before
  [pem-cert]
  (-> pem-cert pem->x509-cert .getNotBefore ->tempus))

(defn not-after
  [pem-cert]
  (-> pem-cert pem->x509-cert .getNotAfter ->tempus))

(defn expires
  [pem-cert]
  (let [x509 (-> pem-cert pem->x509-cert)]
    (->> (i/interval (-> x509 .getNotBefore ->tempus)
                     (-> x509 .getNotAfter ->tempus))
         (i/into :days)
         Math/floor
         int)))

(defn serial-number
  [pem-cert]
  (-> pem-cert pem->x509-cert .getSerialNumber str))


;;; Private

(defn- ^X509CertImpl unsigned-certificate
  ([^X500Name issuer ^PublicKey public-key ^X500Name subject expires-in]
   (unsigned-certificate issuer public-key subject expires-in nil))
  ([^X500Name issuer ^PublicKey public-key ^X500Name subject expires-in exts]
   (let [validity (CertificateValidity. (Date.)
                                        (Date. (long (t/into :long (t/+ (t/now) (d/days expires-in))))))
         extensions (certificate-extensions exts)
         cert-info (doto (X509CertInfo.)
                     (.set X509CertInfo/VERSION (CertificateVersion. CertificateVersion/V3))
                     (.set X509CertInfo/SERIAL_NUMBER (CertificateSerialNumber. (BigInteger. 64 (SecureRandom.))))
                     (.set X509CertInfo/ISSUER issuer)
                     (.set X509CertInfo/KEY (CertificateX509Key. public-key))
                     (.set X509CertInfo/SUBJECT subject)
                     (.set X509CertInfo/VALIDITY validity)
                     (.set X509CertInfo/ALGORITHM_ID (-> default-signature-algorithm AlgorithmId/get CertificateAlgorithmId.)))]
     ;; Can't set a nil extensions, so must be out of doto block
     (when extensions
       (.set cert-info X509CertInfo/EXTENSIONS extensions))
     (X509CertImpl. cert-info))))

(defn- ->tempus
  "Currently assuming UTC timezone"
  [^Date d]
  (->> (-> d .toInstant (.atOffset ZoneOffset/UTC))
       (t/from :native)))

(defn- ->pem
  [o]
  (cond
    (instance? X509CertImpl o)  (join "\n" (-> ["-----BEGIN CERTIFICATE-----"]
                                               (into (->> ^X509CertImpl o .getEncoded bytes/to-b64-str (re-seq #".{1,64}")))
                                               (into ["-----END CERTIFICATE-----"])))
    (instance? PKCS10 o) (join "\n" (-> ["-----BEGIN NEW CERTIFICATE REQUEST-----"]
                                        (into (->> ^PKCS10 o .getEncoded bytes/to-b64-str (re-seq #".{1,64}")))
                                        (into ["-----END NEW CERTIFICATE REQUEST-----"])))
    (instance? PrivateKey o) (join "\n" (-> ["-----BEGIN PRIVATE KEY-----"]
                                            (into (->> ^PrivateKey o .getEncoded bytes/to-b64-str (re-seq #".{1,64}")))
                                            (into ["-----END PRIVATE KEY-----"])))
    (instance? PublicKey o) (join "\n" (-> ["-----BEGIN PUBLIC KEY-----"]
                                           (into (->> ^PublicKey o .getEncoded bytes/to-b64-str (re-seq #".{1,64}")))
                                           (into ["-----END PUBLIC KEY-----"])))
    (instance? X509CRLImpl o) (join "\n" (-> ["-----BEGIN X509 CRL-----"]
                                             (into (->> ^X509CRLImpl o .getEncoded bytes/to-b64-str (re-seq #".{1,64}")))
                                             (into ["-----END X509 CRL-----"])))))

(defn- ^X509CertImpl pem->x509-cert
  [^String pem-cert]
  (->> pem-cert
       pem-bytes
       ByteArrayInputStream.
       clojure.java.io/input-stream
       (.generateCertificate (CertificateFactory/getInstance "X.509"))))

(defn- ^PKCS10 pem->pkcs10-csr
  [^String pem-csr]
  (-> pem-csr
      pem-bytes
      bytes
      PKCS10.))

(defn- ^PrivateKey pem->key
  [^String pem-key]
  (->> pem-key
       pem-bytes
       (#(PKCS8EncodedKeySpec. % default-key-algorithm))
       (.generatePrivate (KeyFactory/getInstance default-key-algorithm))))

(defn- ^PublicKey pem->public-key
  [^String pem-key]
  (->> pem-key
       pem-bytes
       (#(PKCS8EncodedKeySpec. % default-key-algorithm))
       (.generatePublic (KeyFactory/getInstance default-key-algorithm))))

(defn- ^X509CRLImpl pem->x509-crl
  [^String pem-crl]
  (->> pem-crl
       pem-bytes
       ByteArrayInputStream.
       clojure.java.io/input-stream
       X509CRLImpl.))

(defn- pem-bytes
  "Function assumes only a single PEM object that is optionally bookended by
  BEGIN/END lines."
  [^String pem]
  (->> pem
       (#(clojure.string/split % #"\n"))
       (drop-while #(clojure.string/starts-with? % "-----BEGIN"))
       (take-while #(not (clojure.string/starts-with? % "-----END")))
       (map clojure.string/trim-newline)
       clojure.string/join
       bytes/from-b64-str))

(defn- key-tuple
  [^String protocol ^String group]
  (let [kpg (doto (KeyPairGenerator/getInstance protocol)
              (.initialize (ECGenParameterSpec. group)))
        key-pair (.generateKeyPair kpg)]
    [(.getPublic key-pair) (.getPrivate key-pair)]))

(defn- ^CertificateExtensions certificate-extensions
  "Takes in a map of certificate extensions (similar to, but not exactly like) keytool.
  These are parsed into a CertificateExtensions object passed to the Java Certificate
  classes.
  Example: {:bc {:ca true} :san {:dns \"google.com\"} :ku [:crl-sign] :eku [:server-auth :client-auth]}

  Supported extensions are BC, EKU, KU, and SAN.  Unupported (currently) are IAN, SIA, and AIA.
  Explicit criticality is not supported, but the default behaviour is leveraged from the *Extension classes

  More will be added as use cases require and sanity permits."
  [m]
  (when-not (empty? m)
    (let [exts (CertificateExtensions.)]
      (->> (into [] m)
           (map (fn [[k v]]
                  (case k
                    :bc (.set exts BasicConstraintsExtension/NAME (BasicConstraintsExtension. (boolean (or (-> v :ca) false))
                                                                                              (int (or (-> v :pathlen) -1))))
                    :eku (.set exts ExtendedKeyUsageExtension/NAME (ExtendedKeyUsageExtension. (doto (java.util.Vector.)
                                                                                                 (.addAll (->> v
                                                                                                               (mapv (fn [usage]
                                                                                                                       (case usage
                                                                                                                         :any-extended-key-usage (ObjectIdentifier/of (str "2.5.29.37.0"))
                                                                                                                         :server-auth (ObjectIdentifier/of (str "1.3.6.1.5.5.7.3.1"))
                                                                                                                         :client-auth (ObjectIdentifier/of (str "1.3.6.1.5.5.7.3.2"))
                                                                                                                         :code-signing (ObjectIdentifier/of (str "1.3.6.1.5.5.7.3.3"))
                                                                                                                         :email-protection (ObjectIdentifier/of (str "1.3.6.1.5.5.7.3.4"))
                                                                                                                         :ipsec-end-system (ObjectIdentifier/of (str "1.3.6.1.5.5.7.3.5"))
                                                                                                                         :ipsec-tunnel (ObjectIdentifier/of (str "1.3.6.1.5.5.7.3.6"))
                                                                                                                         :ipsec-user (ObjectIdentifier/of (str "1.3.6.1.5.5.7.3.7"))
                                                                                                                         :time-stamping (ObjectIdentifier/of (str "1.3.6.1.5.5.7.3.8"))
                                                                                                                         :ocsp-signing (ObjectIdentifier/of (str "1.3.6.1.5.5.7.3.9"))))))))))

                    :ku (let [key-usage (KeyUsageExtension.)]
                          (->> v
                               (map (fn [usage]
                                      (case usage
                                        :crl-sign (.set key-usage KeyUsageExtension/CRL_SIGN true)
                                        :data-encipherment (.set key-usage KeyUsageExtension/DATA_ENCIPHERMENT true)
                                        :decipher-only (.set key-usage KeyUsageExtension/DECIPHER_ONLY true)
                                        :digital-signature (.set key-usage KeyUsageExtension/DIGITAL_SIGNATURE true)
                                        :encipher-only (.set key-usage KeyUsageExtension/ENCIPHER_ONLY true)
                                        :key-agreement (.set key-usage KeyUsageExtension/KEY_AGREEMENT true)
                                        :key-certsign (.set key-usage KeyUsageExtension/KEY_CERTSIGN true)
                                        :key-encipherment (.set key-usage KeyUsageExtension/KEY_ENCIPHERMENT true)
                                        :non-repudiation (.set key-usage KeyUsageExtension/NON_REPUDIATION true))))
                               dorun)
                          (.set exts KeyUsageExtension/NAME key-usage))
                    :san (let [general-names (GeneralNames.)]
                           (->> (into [] v)
                                (map (fn [[san-k san-v]]
                                       (.add general-names (case san-k
                                                             :email (GeneralName. (RFC822Name. (str san-v)))
                                                             :uri (GeneralName. (URIName. (str san-v)))
                                                             :dns (GeneralName. (DNSName. (str san-v) true))
                                                             :ip (GeneralName. (IPAddressName. (str san-v)))
                                                             :oid (GeneralName. (OIDName. (str san-v)))))))
                                dorun)
                           (.set exts SubjectAlternativeNameExtension/NAME (SubjectAlternativeNameExtension. general-names))))))
           dorun)
      exts)))

(defn- ^X509CertInfo x509-cert-info
  [^X509CertImpl cert]
  (.get cert (str X509CertImpl/NAME "." X509CertImpl/INFO)))
