;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns fides.certs
  (:require [fides.util.keytool :as keytool]
            [fides.util.bytes :as bytes]
            [crusta.core :as sh]
            [utilis.fs :refer [with-temp]]
            [clojure.java.io :as io]
            [clojure.string :refer [join trim]]))

(defn rand-str [len]
  (apply str (take len (repeatedly #(char (+ (rand 26) 65))))))

(defn load-as-base64
  [f]
  (bytes/to-b64-str (with-open [out (java.io.ByteArrayOutputStream.)]
                      (clojure.java.io/copy (clojure.java.io/input-stream f) out)
                      (.toByteArray out))))

(defn generate-private-key
  [alias common-name keystore key-password store-password expires]
  (keytool/run :genkeypair :validity (str expires)
               :alias [alias] :dname [(str "cn=" common-name)]
               :keyalg "EC" :groupname "secp384r1"
               :ext "bc:c" :keystore keystore :keypass key-password :storepass store-password))

(defn generate-root-ca-certificate
  [alias keystore store-password expires]
  (keytool/run :exportcert :validity (str expires) :rfc :keystore keystore :alias [alias] :storepass store-password))

(defn generate-issuing-ca-certificate
  [alias keystore store-password root-alias root-keystore root-store-password expires]
  (->> (sh/pipe (keytool/exec :keystore keystore :storepass store-password :certreq :alias [alias])
                (keytool/exec :keystore root-keystore :storepass root-store-password
                              :gencert :validity (str expires) :alias [root-alias] :ext "bc=0" :ext "san=dns:ca" :rfc))
       sh/stdout-seq
       (join "\n")))

(defn convert-jks-to-p12
  [jks-filename p12-filename jks-password p12-password]
  (keytool/run :importkeystore :srckeystore jks-filename :srcstorepass jks-password
               :deststoretype "PKCS12" :destkeystore p12-filename :deststorepass p12-password))

(defn generate-user-keystore
  [user]
  (let [name (:name user)
        expires (or (:expires user) 365)
        user-pwd "password"]
    (with-temp [user-ks-file user-p12-file user-key-file]
      (generate-private-key name name (.getAbsolutePath user-ks-file) user-pwd user-pwd expires)
      (convert-jks-to-p12 (.getAbsolutePath user-ks-file) (.getAbsolutePath user-p12-file) user-pwd user-pwd)
      @(sh/run ["bash -c" ["openssl" "pkcs12" "-in" user-ks-file "-password" (str "pass:" user-pwd) "-nodes" "-nocerts" "-out" user-key-file]])
      (let [user-crt (generate-root-ca-certificate name (.getAbsolutePath user-ks-file) user-pwd 365)
            user-key (slurp user-key-file)]
        (merge user {:crt (join "\n" [user-crt user-key])})))))

(defn private-keystore
  [alias keypass expires]
  (with-temp [ks-file]
    (keytool/run :genkeypair :alias [alias] :dname [(str "cn=" alias)] :validity (str expires)
                 :keyalg "EC" :groupname "secp384r1" :ext "bc:c"
                 :keystore (.getAbsolutePath ks-file) :keypass keypass :storepass keypass)
    (with-open [i (io/input-stream ks-file)
                o (java.io.ByteArrayOutputStream.)]
      (io/copy i o)
      (.toByteArray o))))

(defn private-key
  [ks keypass]
  (with-temp [ks-file ks-p12-file]
    (with-open [w (io/output-stream (.getAbsolutePath ks-file))] (.write w ks))
    (convert-jks-to-p12 (.getAbsolutePath ks-file) (.getAbsolutePath ks-p12-file) keypass keypass)
    @(sh/run ["bash -c" ["openssl" "pkcs12" "-in" (.getAbsolutePath ks-file)
                         "-password" (str "pass:" keypass)
                         "-nodes" "-nocerts"]])))

(defn generate-crl
  ([ca-crt ca-ks ca-pwd valid]
   (generate-crl ca-crt ca-ks ca-pwd valid []))
  ([ca-crt ca-ks ca-pwd valid crts]
   (with-temp [ca-crt-file ca-key-file]
     @(sh/run ["mkdir" "-p" "ca"])
     @(sh/run ["touch" "ca/index.txt"])
     (spit ca-crt-file ca-crt)
     (spit ca-key-file (private-key (bytes/from-b64s (.getBytes ca-ks)) ca-pwd))
     (doseq [crt crts]
       (try
         (with-temp [crt-file]
           (spit crt-file crt)
           @(sh/run ["openssl ca" "-revoke" (.getAbsolutePath crt-file)
                     "-config" "openssl.cnf"
                     "-cert" (.getAbsolutePath ca-crt-file)
                     "-keyfile" (.getAbsolutePath ca-key-file)]))
         (catch Exception e (println e))))
     (let [crl @(sh/run ["openssl ca" "-gencrl" "-crldays" (str valid)
                         "-config" "openssl.cnf"
                         "-cert" (.getAbsolutePath ca-crt-file)
                         "-keyfile" (.getAbsolutePath ca-key-file)])]
       @(sh/run ["rm" "ca/index.txt"])
       @(sh/run ["rmdir" "ca"])
       crl))))

(defn generate-trust-chain
  [name valid target-dir]
  (with-temp [root-ks-file root-key-file root-crt-file issuing-crt-file]
    (let [root-alias "root"
          root-cn (str "\"" name " Root\"")
          root-pwd (rand-str 24)
          issuing-alias "issuing-ca"
          issuing-cn (str "\"" name " Issuing CA\"")
          issuing-ks-file (io/file (str (or target-dir ".") "/issuing-ca.jks"))
          issuing-pwd (rand-str 24)]
      (generate-private-key root-alias root-cn (.getAbsolutePath root-ks-file) root-pwd root-pwd valid)
      (let [root-crt (generate-root-ca-certificate root-alias (.getAbsolutePath root-ks-file) root-pwd valid)
            root-crl (generate-crl root-crt (load-as-base64 (.getAbsolutePath root-ks-file)) root-pwd valid)]
        (generate-private-key issuing-alias issuing-cn (.getAbsolutePath issuing-ks-file) issuing-pwd issuing-pwd valid)
        (let [issuing-ks (load-as-base64 (.getAbsolutePath issuing-ks-file))
              issuing-crt (generate-issuing-ca-certificate issuing-alias (.getAbsolutePath issuing-ks-file) issuing-pwd root-alias (.getAbsolutePath root-ks-file) root-pwd valid)]

          ;; Temporary block - store files required for signing service certs to use
          ;; Eventual target is to do it through CSR API
          (if (not (empty? target-dir))
            (do (spit (str target-dir "/root.crt") root-crt)
                (spit (str target-dir "/issuing-ca.crt") issuing-crt)
                (spit (str target-dir "/issuing-ca.pwd") issuing-pwd))
            (.delete issuing-ks-file))

          {:payload (join "\n" [root-crt issuing-crt])
           :root-crt root-crt
           :root-crl root-crl
           :issuing-pwd issuing-pwd
           :issuing-jks issuing-ks
           :issuing-crt issuing-crt})))))

(defn csr
  [alias ks keypass]
  (with-temp [ks-file]
    (with-open [w (io/output-stream (.getAbsolutePath ks-file))] (.write w ks))
    (keytool/run :certreq :alias [alias] :keystore (.getAbsolutePath ks-file) :storepass keypass)))

(defn signed-certificate
  [issuing-ks issuing-keypass csr expiry]
  (with-temp [issuing-ks-file csr-file]
    (with-open [w (io/output-stream (.getAbsolutePath issuing-ks-file))] (.write w issuing-ks))
    (spit (.getAbsolutePath csr-file) csr)
    (keytool/run :gencert :alias "issuing-ca" :validity (str expiry)
                 :keystore (.getAbsolutePath issuing-ks-file) :storepass issuing-keypass
                 :infile (.getAbsolutePath csr-file)
                 :ext "ku:c=dig,keyEnc" :ext "eku=sa,ca" :ext "san=dns:localhost,ip:10.17.79.1" :rfc)))

(defn signed-user-certificate
  [userid issuing-ks issuing-keypass csr expiry]
  (with-temp [issuing-ks-file csr-file]
    (with-open [w (io/output-stream (.getAbsolutePath issuing-ks-file))] (.write w issuing-ks))
    (spit (.getAbsolutePath csr-file) csr)
    (keytool/run :gencert :alias "issuing-ca" :validity (str expiry)
                 :keystore (.getAbsolutePath issuing-ks-file) :storepass issuing-keypass
                 :infile (.getAbsolutePath csr-file)
                 :ext "ku:c=dig,keyEnc" :ext "eku=sa,ca" :ext (str "san=dns:" userid) :rfc)))
