;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns fides.cipher
  "Encrypt and decrypt arbitrary bytes using the AES's CBC block cipher algorithm.
  Integrity is checked with HMAC SHA-256"
  (:require [fides.util.bytes :as bytes]
            [fides.nonce :as nonce]
            [fides.mac :as mac]
            [fides.key :as key])
  (:import java.nio.ByteBuffer
           javax.crypto.Cipher
           javax.crypto.spec.IvParameterSpec))

(defonce ^:private algorithm "AES/CBC/PKCS5PADDING")

(declare extract-crypt-key extract-auth-key
         generate-auth-tag verify-auth-tag)

(def tag-length 16)

(defn encrypt
  [^bytes input {:keys [key iv aad]}]
  (let [ciphertext (-> (doto (Cipher/getInstance algorithm)
                         (.init Cipher/ENCRYPT_MODE
                                (key/from-bytes (extract-crypt-key key))
                                (IvParameterSpec. iv)))
                       (.doFinal input))
        tag (generate-auth-tag ciphertext {:alg :sha256
                                           :auth-key (extract-auth-key key)
                                           :aad aad
                                           :iv iv})]
    (bytes/concat ciphertext tag)))

(defn decrypt
  [^bytes input {:keys [key iv aad] :as params}]
  (let [input-length (count input)
        auth-tag (bytes/slice input (- input-length tag-length) input-length)
        ciphertext (bytes/slice input 0 (- input-length tag-length))
        authkey (extract-auth-key key)]
    (when-not (verify-auth-tag auth-tag ciphertext (assoc params
                                                          :alg :sha256
                                                          :auth-key authkey))
      (throw (ex-info "Message seems corrupt or manipulated."
                      {:type :validation :cause :auth-tag})))
    (-> (doto (Cipher/getInstance algorithm)
          (.init Cipher/DECRYPT_MODE
                 (key/from-bytes (extract-crypt-key key))
                 (IvParameterSpec. iv)))
        (.doFinal ciphertext))))

;;; Private

(defn- extract-crypt-key
  [^bytes key]
  (bytes/slice key 16 32))

(defn- extract-auth-key
  [^bytes key]
  (bytes/slice key 0 16))

(defn- aad->bytes
  [aad]
  (let [length (* (count aad) 8)
        buffer (ByteBuffer/allocate 8)]
    (.putLong buffer length)
    (.array buffer)))

(defn- prep-tag-bytes
  [^bytes input {:keys [iv aad]}]
  (bytes/concat aad iv input (if aad
                               (aad->bytes aad)
                               (byte-array 0))))

(defn- generate-auth-tag
  [^bytes input {:keys [auth-key] :as params}]
  (mac/hash (prep-tag-bytes input params) auth-key tag-length))

(defn- verify-auth-tag
  [^bytes tag ^bytes input {:keys [auth-key] :as params}]
  (mac/verify tag (prep-tag-bytes input params) auth-key tag-length))
