(ns fides.keystore
  (:require [clojure.java.io :as io]
            [clojure.data.codec.base64 :as base64]
            [utilis.fs :refer [rm]]
            [fides.certificates :refer [->pem]])
  (:import [java.io File BufferedOutputStream]
           [java.security KeyStore]))

(declare ^KeyStore base64-string->key-store ks-entry)

(defn entry
  "This function assumes the keystore password [ks-pwd] is also the keystore entry password"
  [^String b64-ks ^String alias ^String ks-pwd & {:keys [temp-file-directory]}]
  (let [ks (base64-string->key-store b64-ks ks-pwd :temp-file-directory temp-file-directory)]
    (ks-entry ks alias ks-pwd)))

(defn ->map
  [^String b64-ks ^String ks-pwd & {:keys [temp-file-directory]}]
  (let [ks (base64-string->key-store b64-ks ks-pwd :temp-file-directory temp-file-directory)]
    (->> (-> ks .aliases enumeration-seq vec)
         (map (fn [alias]
                {alias (ks-entry ks alias ks-pwd)}))
         (reduce merge))))


;;; Private

(defn- base64-string->key-store
  "s is expected to be a base 64 encoded string of a JKS"
  [ks-str ks-pwd & {:keys [temp-file-directory]}]
  (let [tmp-ks-file (io/file (str (when temp-file-directory (str temp-file-directory "/")) (gensym)))]
    (with-open [w (io/output-stream (.getAbsolutePath tmp-ks-file))]
      (.write w (bytes (base64/decode (.getBytes ^String ks-str)))))
    (let [ks (KeyStore/getInstance tmp-ks-file (char-array ks-pwd))]
      (rm tmp-ks-file)
      ks)))

(defn- ks-entry
  [^KeyStore ks ^String alias ^String ks-pwd]
  {:certificate (->pem (.getCertificate ks alias))
   :key (->pem (.getKey ks alias (char-array ks-pwd)))})
