;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns fides.util.bytes
  (:refer-clojure :exclude [concat])
  (:import java.util.Arrays
           java.util.Base64))

(defn to-str
  [^bytes data]
  (String. data "UTF-8"))

(defn from-str
  [^String s]
  (.getBytes s "UTF-8"))

(defn to-b64-str
  [^bytes data & without-padding]
  (.encodeToString (cond-> (java.util.Base64/getEncoder)
                     without-padding (.withoutPadding)) data))

(defn from-b64-str
  [^String s]
  (.decode (Base64/getDecoder) s))

(defn to-b64s
  [^bytes data & without-padding]
  (.encode (cond-> (java.util.Base64/getEncoder)
             without-padding (.withoutPadding)) data))

(defn from-b64s
  [^bytes data]
  (.decode (Base64/getDecoder) data))

(defn to-url-b64s
  [^bytes data & without-padding]
  (.encode (cond-> (Base64/getUrlEncoder)
             without-padding (.withoutPadding)) data))

(defn from-url-b64s
  [^bytes data]
  (.decode (Base64/getUrlDecoder) data))

(defn slice
  [^bytes input ^long start ^long end]
  (Arrays/copyOfRange input start end))

(defn concat
  [& parts]
  (byte-array
   (for [ar parts
         i ar]
     i)))

(defn equals?
  "Test whether two sequences of characters or bytes are equal in a way that
  protects against timing attacks. Note that this does not prevent an attacker
  from discovering the *length* of the data being compared."
  [a b]
  (let [a (map int a), b (map int b)]
    (if (and a b (= (count a) (count b)))
      (zero? (reduce bit-or 0 (map bit-xor a b)))
      false)))
