;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns fides.util.zip
  (:import java.io.ByteArrayInputStream
           java.io.ByteArrayOutputStream
           java.util.zip.Deflater
           java.util.zip.DeflaterOutputStream
           java.util.zip.InflaterInputStream
           java.util.zip.Inflater
           java.util.zip.ZipException))

(defn compress
  ([^bytes input]
   (compress input nil))
  ([^bytes input {:keys [nowrap] :or {nowrap true}}]
   (let [out-stream (ByteArrayOutputStream.)
         defl (Deflater. Deflater/DEFLATED ^Boolean nowrap)]
     (with-open [defl-out-stream (DeflaterOutputStream. out-stream defl)]
       (.write defl-out-stream input))
     (.toByteArray out-stream))))

(defn expand
  ([^bytes input]
    (expand input nil))
  ([^bytes input {:keys [nowrap buffer-size] :or {nowrap true buffer-size 2048}}]
   (let [buffer (byte-array (int buffer-size))
         out-stream (ByteArrayOutputStream.)
         infl (Inflater. ^Boolean nowrap)]
     (try
       (with-open [in-stream (ByteArrayInputStream. input)
                   infl-in-stream (InflaterInputStream. in-stream infl)]
         (loop []
           (let [x (.read infl-in-stream buffer)]
             (when (pos? x)
               (.write out-stream buffer 0 x)
               (recur)))))
       (.toByteArray out-stream)
       (catch ZipException e
         (if nowrap
           (expand input {:nowrap false :buffer-size buffer-size})
           (throw e)))))))
