;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns fides.jws
  (:require [clojure.string :as st]
            [jsonista.core :as j]
            [fides.util.bytes :as bytes]
            [fides.mac :as mac]))

(declare encode-header decode-header
         encode-payload decode-payload
         calculate-signature verify-signature
         split-message)

(defn sign
  "Sign arbitrary length string using json web token/signature."
  [^String input ^bytes key & [{:keys [header]}]]
  (let [header (-> header
                   (assoc :alg :hs256)
                   (encode-header))
        payload (encode-payload input)
        signature (calculate-signature {:key key
                                        :header header
                                        :payload payload})]
    (st/join "." [header payload signature])))

(defn unsign
  "Given a signed message, verify it and return the decoded payload."
  [^String input ^bytes key]
  (let [[header payload signature] (split-message input)]
    (when-not
        (try
          (verify-signature {:key key
                             :signature signature
                             :header header
                             :payload payload})
          (catch java.security.SignatureException se
            (throw (ex-info "Message seems corrupt or manipulated."
                            {:type :validation :cause :signature}
                            se))))
      (throw (ex-info "Message seems corrupt or manipulated."
                      {:type :validation :cause :signature})))
    (decode-payload payload)))


;;; Private

(defn- encode-header
  [header]
  (-> header
      (update :alg #(st/upper-case (name %)))
      (j/write-value-as-string)
      (bytes/from-str)
      (bytes/to-url-b64s :without-padding)
      (bytes/to-str)))

(defn- decode-header
  [^String data]
  (try
    (let [header (-> data
                     (bytes/from-b64-str)
                     (bytes/from-url-b64s)
                     (bytes/to-str)
                     (j/read-value j/keyword-keys-object-mapper))]
      (when-not (map? header)
        (throw (ex-info "Message seems corrupt or manipulated."
                        {:type :validation :cause :header})))
      (update header :alg #(keyword (st/lower-case %))))
    (catch java.lang.NullPointerException e
      (throw (ex-info "Message seems corrupt or manipulated."
                      {:type :validation :cause :header})))
    (catch com.fasterxml.jackson.core.JsonParseException e
      (throw (ex-info "Message seems corrupt or manipulated."
                      {:type :validation :cause :header})))))

(defn- encode-payload
  [^String payload]
  (-> payload
      (bytes/from-str)
      (bytes/to-url-b64s :without-padding)
      (bytes/to-str)))

(defn- decode-payload
  [^String payload]
  (-> payload
      (bytes/from-str)
      (bytes/from-url-b64s)
      (bytes/to-str)))

(defn- calculate-signature
  "Given the bunch of bytes, a private key and algorithm,
  return a calculated signature as byte array."
  [{:keys [key header payload]}]
  (-> (st/join "." [header payload])
      (bytes/from-str)
      (mac/hash key)
      (bytes/to-url-b64s :without-padding)
      (bytes/to-str)))

(defn- verify-signature
  "Given a bunch of bytes, a previously generated
  signature, the private key and algorithm, return
  signature matches or not."
  [{:keys [signature key header payload]}]
  (-> signature
      (bytes/from-str)
      (bytes/from-url-b64s)
      (mac/verify (bytes/from-str (st/join "." [header payload])) key)))

(defn- split-message
  [message]
  (st/split message #"\." 3))
