;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns fides.jws
  (:require [jsonista.core :as json]
            [clojure.string :as st])
  (:import [javax.crypto Mac]
           [javax.crypto.spec SecretKeySpec]
           [java.util Base64]))

(declare encode mac-hash)

(defn sign
  "Sign arbitrary length string using json web token/signature."
  [^String input ^bytes key & [{:keys [header]}]]
  (let [header (-> (assoc header :alg "HS256")
                   json/write-value-as-string
                   encode)
        payload (encode input)
        signature (-> (st/join "." [header payload])
                      (encode (partial mac-hash key)))]
    (st/join "." [header payload signature])))


;;; Private

(defn- encode ^String
  [^String s & [bytes-fn]]
  (-> (.getBytes s "UTF-8")
      (cond-> bytes-fn bytes-fn)
      (->> (.encode (.withoutPadding (Base64/getUrlEncoder))))
      (String. "UTF-8")))

(def ^:private algorithm "HmacSHA256")

(defn mac-hash
  [^bytes key ^bytes data]
  (-> (doto (Mac/getInstance algorithm)
        (.init (SecretKeySpec. key algorithm)))
      (.doFinal data)))
