;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns fides.hash
  (:require [fides.random :as random]
            [utilis.base64 :as base64]
            [clojure.java.io :as io])
  (:import  [org.bouncycastle.crypto.generators Argon2BytesGenerator]
            [org.bouncycastle.crypto.params Argon2Parameters$Builder]
            [java.security MessageDigest DigestInputStream]
            [java.io File InputStream]))

(declare digest)

(defn argon2
  ([v] (argon2 (random/bytes 16) v))
  ([salt v] (argon2 salt v {}))
  ([salt v {:keys [iterations memory-kb parallelism]
            :or {iterations 3
                 memory-kb 16
                 parallelism 4}}]
   (let [v-b (if (bytes? v) v (.getBytes ^String v))
         salt-b (if (bytes? salt) salt (.getBytes ^String salt))
         builder (doto (Argon2Parameters$Builder.)
                   (.withIterations iterations)
                   (.withMemoryAsKB (* memory-kb 1024))
                   (.withParallelism parallelism)
                   (.withSalt salt-b))
         parameters (.build builder)
         hash-b (byte-array 32)]
     (doto (Argon2BytesGenerator.)
       (.init parameters)
       (.generateBytes v-b hash-b))
     [(base64/encode salt-b)
      (base64/encode hash-b)])))

(defn sha512
  [v]
  (digest "SHA-512" v))

(defn sha256
  [v]
  (digest "SHA-256" v))

(defn md5
  [v]
  (digest "MD5" v))


;;; Private

(defn- compute
  [algorithm v]
  (let [digest (MessageDigest/getInstance algorithm)]
    (cond
      (instance? InputStream v)
      (with-open [dis (DigestInputStream. v digest)]
        (while (pos? (.read dis)))
        (.digest digest))

      (instance? File v)
      (with-open [fis (io/input-stream v)]
        (compute algorithm fis))

      (bytes? v)
      (.digest digest v)

      (string? v)
      (compute algorithm (.getBytes ^String v)))))

(defn- bytes->string
  [bytes]
  (let [sb (StringBuilder.)]
    (doseq [b (vec bytes)]
      (.append sb (format "%02x" b)))
    (.toString sb)))

(defn- digest
  [algorithm v]
  (bytes->string (compute algorithm v)))
