;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns fides.jws
  (:require [jsonista.core :as json]
            [clojure.string :as st])
  (:import  [javax.crypto Mac]
            [javax.crypto.spec SecretKeySpec]
            [java.util Base64]))

(declare encode mac-hash rsa->pk)

(defn sign
  "Sign arbitrary length string using json web token/signature.
  Algorithm default is HS256."
  [^String input ^bytes key & [{:keys [header]}]]
  (let [header (-> {:alg "HS256"} (merge header))
        algo (:alg header)
        encoded-header (-> header json/write-value-as-string encode)
        encoded-payload (encode input)

        encoded-signature (case algo
                            "HS256" (-> (st/join "." [encoded-header encoded-payload])
                                        (encode #(mac-hash key % "HmacSHA256")))
                            "RS256" (->> (doto (Signature/getInstance "SHA256withRSA")
                                           (.initSign (ssh/rsa-private-key->pkcs8 (String. key "UTF-8")))
                                           (.update (.getBytes (st/join "." [encoded-header encoded-payload]) "UTF-8")))
                                         .sign
                                         (.encode (.withoutPadding (Base64/getUrlEncoder)))
                                         (#(String. ^bytes % "UTF-8"))))]
    (st/join "." [encoded-header encoded-payload encoded-signature])))


;;; Private

(defn- encode ^String
  [^String s & [bytes-fn]]
  (let [^"[B" bytes (cond-> (.getBytes s "UTF-8")
                      bytes-fn bytes-fn)]
    (-> (.encode (.withoutPadding (Base64/getUrlEncoder)) bytes)
        (String. "UTF-8"))))

(defn- mac-hash
  [^bytes key ^bytes data ^String algorithm]
  (-> (doto (Mac/getInstance algorithm)
        (.init (SecretKeySpec. key algorithm)))
      (.doFinal data)))
