;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns file-watch-clj.core
  (:import [java.io File]
           [java.nio.file FileSystems Paths StandardWatchEventKinds WatchEvent$Kind WatchEvent WatchKey WatchService]))

(defn watch
  [{:keys [paths handler]}]
  (try
    (future
      (try
        (let [ws ^WatchService (.newWatchService (FileSystems/getDefault))]
          ;; register paths for watching
          (->> paths
               (map #(doto (Paths/get % (into-array [""]))
                       (.register ws (into-array WatchEvent$Kind
                                                 [StandardWatchEventKinds/ENTRY_CREATE
                                                  StandardWatchEventKinds/ENTRY_MODIFY
                                                  StandardWatchEventKinds/ENTRY_DELETE]))))
               doall)

          ;; process events
          (loop [watch-key ^WatchKey (.take ws)]
            (->> (.pollEvents watch-key)
                 (map (fn [^WatchEvent we]
                        (let [e {:count (.count we)
                                 :kind (get {StandardWatchEventKinds/ENTRY_CREATE :create
                                             StandardWatchEventKinds/ENTRY_MODIFY :modify
                                             StandardWatchEventKinds/ENTRY_DELETE :delete}
                                            (.kind we) :unknown)
                                 :file ^File (clojure.java.io/file (str (-> watch-key .watchable .toString) "/" (.context we)))}]
                          (handler e))))
                 doall)
            (.reset watch-key)
            (recur ^WatchKey (.take ws))))
        (catch Exception e
          (println (str "[file-watch-clj] Exception in future watching files: " e))
          (println (map #(str "\n" %1) (.getStackTrace e))))))
    (catch Exception e
      (println (str "[file-watch-clj] Exception watching files: " e))
      (println (map #(str "\n" %1) (.getStackTrace e))))))

(defn close
  [w]
  (future-cancel w))
