;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns fluxus.actor
  (:require #?(:clj [fluxus.fiber :refer [fiber]])
            #?(:clj  [clojure.core.async :as a]
               :cljs [cljs.core.async :as a])
            [clojure.core.match :refer [match]]
            [utilis.coll :refer [nth-last]]
            [utilis.string :as ust])
  #?(:clj (:import [clojure.lang IObj IMeta]
                   [java.util UUID]
                   [java.lang IllegalArgumentException])))

(declare pr-actor)

(defprotocol IActor
  (! [a msg]))

(deftype Actor [fiber mailbox messages links meta-map]
  Object
  (toString
    [^Actor this]
    (pr-actor this))
  #?(:cljs IHash)
  (#?(:clj hashCode :cljs -hash)
    [_]
    (hash [:fluxus/actor mailbox]))
  #?(:cljs IEquiv)
  (#?(:clj equals :cljs -equiv)
    [^Actor this other]
    (boolean
     (when (instance? Actor other)
       (= (#?(:clj .mailbox :cljs (.-mailbox this)) this)
          (#?(:clj .mailbox :cljs (.-mailbox this)) ^Actor other)))))

  #?(:clj IObj :cljs IWithMeta)
  (#?(:clj withMeta :cljs -with-meta)
    [_ meta-map]
    (Actor. fiber mailbox messages links meta-map))

  IMeta
  (#?(:clj meta :cljs -meta)
    [_]
    meta-map)

  IActor
  (! [this msg]
    (#?(:clj a/>!! :cljs a/>!) mailbox [msg])
    msg)

  #?@(:cljs
      [IPrintWithWriter
       (-pr-writer [this w _opts] (write-all w (pr-actor this)))]))

#?(:clj
   (defmethod print-method Actor [^Actor p w]
     (.write ^java.io.Writer w ^String (pr-actor p))))

(def ^:dynamic ^Actor *self*)

(defn self
  []
  *self*)

(defmacro spawn
  [& body]
  (let [opt-keys #{:name :link :trap :mailbox-size :overflow-policy}
        opts (->> body
                  (partition-all 2)
                  (map vec)
                  (take-while (fn [[k _ :as pair]]
                                (and (= 2 (count pair))
                                     (get opt-keys k))))
                  (into {}))
        body (drop (* 2 (count opts)) body)
        mailbox-size (get opts :mailbox-size 1024)]
    `(let [mailbox# (a/chan ~mailbox-size)
           messages# (atom [])
           self# (Actor. (atom nil) mailbox# messages#
                         (atom ~(if-let [link (get opts :link)]
                                  #{link}
                                  #{})) nil)
           fiber# (#?(:clj fiber :cljs a/go)
                   (binding [*self* self#]
                     ~@body))]
       (reset! (#?(:clj .fiber :cljs .-fiber) self#) fiber#)
       self#)))

(defmacro spawn-link
  [& body]
  `(if-not (boolean (instance? Actor (self)))
     (throw (ex-info "must be called from actor" {}))
     (spawn :link (self) ~@body)))

;; TODO: Attempting to link to a non-existent process kills the linker
(defn link!
  ([other]
   (link! (self) other))
  ([^Actor actor0 ^Actor actor1]
   (swap! (#?(:clj .links :cljs .-links) actor0) conj actor1)
   (swap! (#?(:clj .links :cljs .-links) actor1) conj actor0)
   nil))

(defn unlink!
  ([other]
   (unlink! (self) other))
  ([^Actor actor0 ^Actor actor1]
   (swap! (#?(:clj .links :cljs .-links) actor0) disj actor1)
   (swap! (#?(:clj .links :cljs .-links) actor1) disj actor0)
   nil))

(defn maketag
  []
  (str #?(:clj (UUID/randomUUID) :cljs (random-uuid))))

(defmacro receive
  [& body]
  (let [[body after] (if (and (>= (count body) 3)
                              (= :timeout (nth-last body 2)))
                       (split-at (- (count body) 3) body)
                       [body nil])
        mailbox (gensym "mailbox_")
        result (gensym "result_")]
    `(loop []
       (let [messages# (#?(:clj .messages :cljs .-messages) *self*)
             ~result (atom ::no-match)
             match# (fn [msg#]
                      (try
                        (reset! ~result (match (first msg#) ~@body))
                        true
                        (catch IllegalArgumentException e#
                          (if (= (.getMessage e#)
                                 (str "No matching clause: " (first msg#)))
                            false
                            (throw e#)))))]
         (swap!
          messages#
          (fn [messages#]
            (->> (range 0 (count messages#))
                 (map (fn [i#] (drop i# messages#)))
                 (reduce (fn [seen# [msg# & rest#]]
                           (if (match# msg#)
                             (reduced (into seen# rest#))
                             (conj seen# msg#))) []))))
         (when (= ::no-match @~result)
           (let [~mailbox (#?(:clj .mailbox :cljs .-mailbox) *self*)
                 new-msg# ~(if after
                             `(let [timeout-ch# (a/timeout ~(second after))
                                    [v# ch#] (#?(:clj a/alts!! :cljs a/alts!)
                                              [~mailbox timeout-ch#])]
                                (if (= timeout-ch# ch#)
                                  (do (reset! ~result ~(last after)) nil)
                                  v#))
                             `(#?(:clj a/<!! :cljs a/<!) ~mailbox))]
             (when (and new-msg# (not (match# new-msg#)))
               (swap! messages# conj new-msg#))))
         (if (= ::no-match @~result)
           (recur)
           @~result)))))


;;; Private

(defn- pr-actor
  [^Actor a]
  (ust/format (str "#<fluxus/actor@" #?(:clj "0x%x" :cljs "%s") " [%d]>")
              (hash a)
              (count @(#?(:clj .messages :cljs .-messages) a))))
