(ns formal.input
  (:require [formal.form :as form]
            [malli.core :as m]
            [malli.transform :as mt]
            [reagent.core :as r]
            [react :as react]
            [utilis.fn :refer [fsafe]]
            [utilis.js :as j]))

(declare reg-input use-validator use-decoder
         input-change-handler key-down-handler)

(defn -input
  [props & children]
  (let [registry (react/useContext form/FormContext)
        input-state (reg-input registry props)
        validator (use-validator props)
        decoder (use-decoder props)]
    (r/as-element
     (into [:input (-> props
                       (dissoc :schema :on-value :on-error)
                       (assoc :on-change (partial input-change-handler
                                            {:props props
                                             :validator validator
                                             :decoder decoder
                                             :input-state input-state})
                              :on-key-down (partial key-down-handler {:props props})))]
           children))))

;;; Private

(defn- reg-input
  [registry props]
  (let [[inputId _] (react/useState (str (random-uuid)))
        [inputState _] (react/useState (r/atom (select-keys props [:name :key :required])))]
    (react/useEffect
     (fn []
       (swap! registry assoc-in [:inputs inputId] inputState)
       (fn [] (swap! registry update :inputs dissoc inputId)))
     #js [])
    (react/useEffect
     (fn []
       (let [key (:key props)]
         (when key
           (swap! registry update :input-key-mappings assoc key inputId))
         (fn []
           (when key
             (swap! registry update :input-key-mappings dissoc key)))))
     #js [(:key props)])
    inputState))

(defn- input-change-handler
  [{:keys [props validator decoder input-state]} event]
  (let [{:keys [on-change on-value on-error]} props
        raw-value (j/get-in event [:target :value])
        value (decoder raw-value)
        error? (when (or (not (string? value))
                         (and value (pos? (count value))))
                 (not (validator value)))]
    (swap! input-state
           (fn [input-state]
             (cond-> (assoc input-state
                            :raw-value raw-value
                            :value value)
               error? (assoc :error error?)
               (not error?) (dissoc :error)
               (:name props) (assoc :name (:name props))
               (:key props) (assoc :key (:key props))
               (:required props) (assoc :required (:required props)))))
    (if error?
      (when (fn? on-error) (on-error value))
      (when (fn? on-value) (on-value value)))
    ((fsafe on-change) event)))

(defn- use-validator
  [props]
  (react/useMemo (fn []
                   (if-let [schema (:schema props)]
                     (m/validator (if (fn? schema)
                                    [:fn schema]
                                    schema))
                     (constantly true)))
                 #js [(:schema props)]))

(defn- use-decoder
  [props]
  (react/useMemo (fn []
                   (if-let [schema (:schema props)]
                     (m/decoder (if (fn? schema)
                                  [:fn schema]
                                  schema)
                                mt/string-transformer)
                     identity))
                 #js [(:schema props)]))

(defn- key-down-handler
  [{:keys [props]} event]
  (let [{:keys [on-key-down]} props]
    (when (= 13 (j/get event :keyCode))
      (doto event
        (j/call :preventDefault)
        (j/call :stopPropagation)))
    ((fsafe on-key-down) event)))
