(ns formal.form
  (:require [reagent.core :as r]
            [react :as react]
            [utilis.js :as j]
            [utilis.fn :refer [fsafe]]))

(def FormContext (react/createContext))

(declare validation-watcher registry-values)

(defn -form
  [props & children]
  (let [context (react/useContext FormContext)
        [registry _] (react/useState (r/atom {:inputs {}
                                              :input-key-mappings {}}))]
    (r/as-element
     [:> (j/get FormContext :Provider) {:value registry}
      (into [:form (-> props
                       (dissoc :on-value :on-error)
                       (update :on-submit (fn [on-submit]
                                            (when on-submit
                                              (fn [event]
                                                (doto event
                                                  (j/call :stopPropagation)
                                                  (j/call :preventDefault))
                                                (let [{:keys [value error? has-all-inputs?]} (registry-values registry)]
                                                  (when (and has-all-inputs? (not error?))
                                                    (on-submit value)))
                                                false)))))
             [:f> validation-watcher
              (-> props
                  (select-keys [:on-value :on-error])
                  (merge {:registry registry}))]]
            children)])))

;;; Private

(defn- registry-values
  [registry]
  (let [inputs (->> @registry
                    :inputs
                    vals
                    (map deref))
        error? (boolean
                (when (seq inputs)
                  (some :error inputs)))
        has-all-inputs? (->> inputs
                             (some (fn [{:keys [required raw-value error]}]
                                     (or error
                                         (and required
                                              (not (seq raw-value))))))
                             not)]
    (merge {:inputs inputs
            :error? error?
            :has-all-inputs? has-all-inputs?}
           (when (some :raw-value inputs)
             {:value inputs}))))

(defn- for-each-submit-button
  [ref f]
  (when-let [node (j/get ref :current)]
    (when-let [form (j/call node :closest "form")]
      (let [buttons (j/call form :getElementsByTagName "button")]
        (doseq [index (range (j/get buttons :length))]
          (let [button (j/get buttons index)]
            (when (= "submit" (j/call button :getAttribute "type"))
              (f button))))))))

(defn- toggle-buttons-disabled
  [registry ref disabled?]
  (react/useEffect
   (fn []
     (for-each-submit-button
      ref (fn [button]
            (if disabled?
              (j/call button :setAttribute "disabled" "")
              (j/call button :removeAttribute "disabled"))))
     (fn []))
   #js [disabled?]))

(defn- validation-watcher
  [{:keys [registry on-error on-value]}]
  (let [ref (react/useRef nil)
        {:keys [inputs value error? has-all-inputs?]} (registry-values registry)]
    (toggle-buttons-disabled registry ref (or error? (not has-all-inputs?)))
    (react/useEffect
     (fn []
       (if error?
         ((fsafe on-error))
         (when (some :raw-value inputs)
           ((fsafe on-value) inputs)))
       (fn []))
     #js [inputs])
    (r/as-element
     [:div {:ref ref}])))
