(ns formal.input
  (:require [formal.form :as form]
            [malli.core :as m]
            [malli.transform :as mt]
            [reagent.core :as r]
            [react :as react]
            [utilis.types.number :refer [string->long
                                         string->double]]
            [utilis.fn :refer [fsafe]]
            [utilis.js :as j]))

(declare reg-input use-validator use-decoder handle-default-value
         input-change-handler key-down-handler checkbox?)

(defn -input
  [{:keys [component] :or {component :input} :as props} & children]
  (let [registry (react/useContext form/FormContext)
        input-state (reg-input registry props)
        validator (use-validator props)
        decoder (use-decoder props)
        change-handler (partial input-change-handler
                          {:props props
                           :validator validator
                           :decoder decoder
                           :input-state input-state})]
    (handle-default-value props input-state change-handler)
    (r/as-element
     (into [component (cond-> props
                        true (dissoc :schema :on-value :on-error :component)
                        true (assoc :on-change change-handler)
                        (= component :input) (assoc :on-key-down (partial key-down-handler {:props props}))
                        (checkbox? props) (assoc :checked (boolean (:value @input-state))))]
           children))))

;;; Private

(defn- safe-keyword
  [s]
  ((fsafe keyword) s))

(defn- reg-input
  [registry props]
  (let [[inputId _] (react/useState (str (random-uuid)))
        [inputState _] (react/useState (r/atom (select-keys props [:name :key :required])))]
    (react/useEffect
     (fn []
       (swap! registry assoc-in [:inputs inputId] inputState)
       (fn [] (swap! registry update :inputs dissoc inputId)))
     #js [])
    (react/useEffect
     (fn []
       (let [key (:key props)]
         (when key
           (swap! registry update :input-key-mappings assoc key inputId))
         (fn []
           (when key
             (swap! registry update :input-key-mappings dissoc key)))))
     #js [(:key props)])
    inputState))

(defn checkbox?
  [props]
  (= :checkbox
     (some-> (:type props)
             keyword)))

(defn- input-change-handler
  [{:keys [props validator decoder input-state]} event]
  (let [{:keys [on-change on-value on-error]} props
        raw-value (if (checkbox? props)
                    (boolean (j/get-in event [:target :checked]))
                    (j/get-in event [:target :value]))
        value (if (string? raw-value)
                (decoder raw-value)
                raw-value)
        empty-value? (and (string? value) (empty? value))
        error? (when (or (not (string? value))
                         (and value (not empty-value?)))
                 (not (validator value)))]
    (swap! input-state
           (fn [input-state]
             (cond-> (assoc input-state
                            :raw-value raw-value
                            :value (when (not empty-value?) value))
               error? (assoc :error error?)
               (not error?) (dissoc :error)
               (j/get event :initial) (assoc :initial true)
               (not (j/get event :initial)) (dissoc :initial)
               (:name props) (assoc :name (:name props))
               (:key props) (assoc :key (:key props))
               (:required props) (assoc :required (:required props)))))
    (if error?
      (when (fn? on-error) (on-error value))
      (when (fn? on-value) (on-value value)))
    ((fsafe on-change) event)))

(defn- use-validator
  [props]
  (react/useMemo (fn []
                   (if-let [schema (:schema props)]
                     (m/validator (if (fn? schema)
                                    [:fn schema]
                                    schema))
                     (constantly true)))
                 #js [(:schema props)]))

(defn- use-decoder
  [props]
  (react/useMemo (fn []
                   (if-let [schema (:schema props)]
                     (if (and (fn? schema)
                              (= :number (safe-keyword (:type props))))
                       (fn [value]
                         (or (when (string? value)
                               (if (re-find #"\." value)
                                 (string->double value)
                                 (string->long value)))
                             value))
                       (m/decoder (if (fn? schema)
                                    [:fn schema]
                                    schema)
                                  mt/string-transformer))
                     identity))
                 #js [(:schema props)]))

(defn- key-down-handler
  [{:keys [props]} event]
  (let [{:keys [on-key-down]} props]
    (when (= 13 (j/get event :keyCode))
      (doto event
        (j/call :preventDefault)
        (j/call :stopPropagation)))
    ((fsafe on-key-down) event)))

(defn- handle-default-value
  [props input-state change-handler]
  (react/useEffect
   (fn []
     (when-let [default-value (:default-value props)]
       (when (not (:value @input-state))
         (change-handler
          (clj->js
           {:target (if (checkbox? props)
                      {:checked default-value}
                      {:value default-value})
            :initial true}))))
     (fn []))
   #js [(:default-value props)]))
