(ns formal.input
  (:require [formal.form :as form]
            [malli.core :as m]
            [malli.transform :as mt]
            [helix.core :refer [$ <> defnc]]
            [helix.dom :refer [div] :as dom]
            [helix.children :as hch]
            [signum.signal :refer [signal alter!]]
            [signum.hooks :refer [use-signal]]
            [react :as react]
            [utilis.types.number :refer [string->long
                                         string->double]]
            [utilis.fn :refer [fsafe]]
            [utilis.js :as j]))

(declare reg-input use-validator use-decoder handle-default-value
         input-change-handler key-down-handler checkbox?)

(defnc input
  [{:keys [component] :or {component :input} :as props}]
  (let [registry (or (react/useContext form/FormContext)
                     (do (js/console.warn "Using a formal/input component outside of a formal.form/FormContext. You should use formal inputs as descendants of a formal.form/form component.")
                         (signal {})))
        input-state (reg-input registry props)
        input-state-value (use-signal input-state)
        validator (use-validator props)
        decoder (use-decoder props)
        change-handler (partial input-change-handler
                          {:props props
                           :validator validator
                           :decoder decoder
                           :input-state input-state})]
    (handle-default-value props input-state change-handler)
    (let [props* (cond-> props
                   true (dissoc :schema :on-value :on-error :component)
                   true (assoc :on-change change-handler)
                   (and (= component :input)
                        (or (nil? (:type props))
                            (#{"text"
                               "email"
                               "number"
                               "password"
                               "search"
                               "submit"
                               "tel"
                               "url"} (:type props)))) (assoc :on-key-down
                                                              (partial key-down-handler
                                                                 {:registry registry
                                                                  :props props}))
                   (checkbox? props) (assoc :checked (boolean (:value input-state-value)))
                   true (dissoc :children))]
      (cond
        (fn? component)
        (component props* (hch/children props))

        (keyword? component)
        ($ (name component) {& props*}
           (hch/children props))

        :else (throw (ex-info "Unrecognized component in formal.input/input" {:component component}))))))

;;; Private

(defn- safe-keyword
  [s]
  ((fsafe keyword) s))

(defn- reg-input
  [registry props]
  (let [[input-id _] (react/useState (str (or (:id props) (random-uuid))))
        [input-state _] (react/useState (signal (-> props
                                                    (select-keys [:name :required])
                                                    (assoc :id input-id))))]
    (react/useEffect
     (fn []
       (alter! registry assoc-in [:inputs input-id] input-state)
       (fn [] (alter! registry update :inputs dissoc input-id)))
     #js [])
    (react/useEffect
     (fn []
       (let [id (:id props)]
         (when id
           (alter! registry update :input-key-mappings assoc id input-id))
         (fn []
           (when id
             (alter! registry update :input-key-mappings dissoc id)))))
     #js [(:id props)])
    input-state))

(defn checkbox?
  [props]
  (= :checkbox
     (some-> (:type props)
             keyword)))

(defn- input-change-handler
  [{:keys [props validator decoder input-state]} event]
  (let [{:keys [on-change on-value on-error]} props
        raw-value (if (checkbox? props)
                    (boolean (j/get-in event [:target :checked]))
                    (j/get-in event [:target :value]))
        value (if (string? raw-value)
                (decoder raw-value)
                raw-value)
        empty-value? (and (string? value) (empty? value))
        error? (when (or (not (string? value))
                         (and value (not empty-value?)))
                 (not (validator value)))]
    (alter! input-state
            (fn [input-state]
              (cond-> (assoc input-state
                             :raw-value raw-value
                             :value (when (not empty-value?) value))
                error? (assoc :error error?)
                (not error?) (dissoc :error)
                (j/get event :initial) (assoc :initial true)
                (not (j/get event :initial)) (dissoc :initial)
                (:name props) (assoc :name (:name props))
                (:id props) (assoc :id (:id props))
                (:required props) (assoc :required (:required props)))))
    (if error?
      (when (fn? on-error) (on-error value))
      (when (fn? on-value) (on-value value)))
    ((fsafe on-change) event)))

(defn- use-validator
  [props]
  (react/useMemo (fn []
                   (if-let [schema (:schema props)]
                     (m/validator (if (fn? schema)
                                    [:fn schema]
                                    schema))
                     (constantly true)))
                 #js [(:schema props)]))

(defn- use-decoder
  [props]
  (react/useMemo (fn []
                   (if-let [schema (:schema props)]
                     (if (and (fn? schema)
                              (= :number (safe-keyword (:type props))))
                       (fn [value]
                         (or (when (string? value)
                               (if (re-find #"\." value)
                                 (string->double value)
                                 (string->long value)))
                             value))
                       (m/decoder (if (fn? schema)
                                    [:fn schema]
                                    schema)
                                  mt/string-transformer))
                     identity))
                 #js [(:schema props)]))

(defn- key-down-handler
  [{:keys [props registry]} event]
  (let [{:keys [on-key-down type]} props
        {:keys [on-submit]} @registry]
    (if on-key-down
      (on-key-down event)
      (when (= 13 (j/get event :keyCode))
        (when (fn? on-submit)
          (on-submit event))))))

(defn- handle-default-value
  [props input-state change-handler]
  (react/useEffect
   (fn []
     (when-let [default-value (:default-value props)]
       (when (not (:value @input-state))
         (change-handler
          (clj->js
           {:target (if (checkbox? props)
                      {:checked default-value}
                      {:value default-value})
            :initial true}))))
     (fn []))
   #js [(:default-value props)]))
